package f.cking.software.ui.main

import android.app.Application
import androidx.annotation.DrawableRes
import androidx.compose.runtime.Composable
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import com.vanpra.composematerialdialogs.MaterialDialogState
import f.cking.software.R
import f.cking.software.data.helpers.BleScannerHelper
import f.cking.software.data.helpers.IntentHelper
import f.cking.software.data.helpers.LocationProvider
import f.cking.software.data.helpers.PermissionHelper
import f.cking.software.data.repo.SettingsRepository
import f.cking.software.service.BgScanService
import f.cking.software.ui.ScreenNavigationCommands
import f.cking.software.ui.devicelist.DeviceListScreen
import f.cking.software.ui.journal.JournalScreen
import f.cking.software.ui.profileslist.ProfilesListScreen
import f.cking.software.ui.settings.SettingsScreen
import f.cking.software.utils.navigation.Router
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.launch

class MainViewModel(
    private val permissionHelper: PermissionHelper,
    private val context: Application,
    private val bluetoothHelper: BleScannerHelper,
    private val settingsRepository: SettingsRepository,
    private val locationProvider: LocationProvider,
    private val intentHelper: IntentHelper,
    private val router: Router,
) : ViewModel() {

    var scanStarted: Boolean by mutableStateOf(BgScanService.state.value.isProcessing())
    var bgServiceIsActive: Boolean by mutableStateOf(BgScanService.isActive)
    var showLocationDisabledDialog: MaterialDialogState = MaterialDialogState()
    var showBluetoothDisabledDialog: MaterialDialogState = MaterialDialogState()

    var tabs by mutableStateOf(
        listOf(
            Tab(
                iconRes = R.drawable.ic_home_outline,
                selectedIconRes = R.drawable.ic_home,
                text = context.getString(R.string.menu_device_list),
                selected = true,
            ) { DeviceListScreen.Screen() },
            Tab(
                iconRes = R.drawable.ic_alert_outline,
                selectedIconRes = R.drawable.ic_alert,
                text = context.getString(R.string.menu_radar_profiles),
                selected = false,
            ) { ProfilesListScreen.Screen() },
            Tab(
                iconRes = R.drawable.ic_journal_outline,
                selectedIconRes = R.drawable.ic_journal,
                text = context.getString(R.string.menu_journal),
                selected = false,
            ) { JournalScreen.Screen() },
            Tab(
                iconRes = R.drawable.ic_settings_outline,
                selectedIconRes = R.drawable.ic_settings,
                text = context.getString(R.string.menu_settings),
                selected = false,
            ) { SettingsScreen.Screen() },
        )
    )

    init {
        observeScanInProgress()
        observeServiceIsLaunched()
    }

    fun onScanButtonClick() {
        checkPermissions {
            BgScanService.scan(context)
        }
    }

    fun onTabClick(tab: Tab) {
        val list = tabs.map { it.copy(selected = it == tab) }
        tabs = list
    }

    fun runBackgroundScanning() {
        checkPermissions {
            if (BgScanService.isActive) {
                BgScanService.stop(context)
            } else if (!locationProvider.isLocationAvailable()) {
                showLocationDisabledDialog.show()
            } else if (!bluetoothHelper.isBluetoothEnabled()) {
                showBluetoothDisabledDialog.show()
            } else {
                BgScanService.start(context)
            }
        }
    }

    fun onTurnOnLocationClick() {
        intentHelper.openLocationSettings()
    }

    fun onTurnOnBluetoothClick() {
        intentHelper.openBluetoothSettings()
    }

    fun needToShowPermissionsIntro(): Boolean {
        return !settingsRepository.getPermissionsIntroWasShown()
    }

    fun userHasPassedPermissionsIntro() {
        settingsRepository.setPermissionsIntroWasShown(true)
    }

    fun needToShowDisclaimer(): Boolean {
        return !settingsRepository.getDisclaimerWasAccepted()
    }

    fun disclaimerWasAccepted() {
        settingsRepository.setDisclaimerWasAccepted(true)
    }

    fun checkAndShowAboutApp() {
        if (!settingsRepository.getWhatIsThisAppForWasShown()) {
            router.navigate(ScreenNavigationCommands.OpenAboutScreen)
            settingsRepository.setWhatIsThisAppForWasShown(true)
        }
    }

    private fun observeScanInProgress() {
        viewModelScope.launch {
            BgScanService.state
                .map { it.isProcessing() }
                .distinctUntilChanged()
                .collect { scanStarted = it }
        }
    }

    private fun observeServiceIsLaunched() {
        viewModelScope.launch {
            BgScanService.observeIsActive()
                .collect { bgServiceIsActive = it }
        }
    }

    private fun checkPermissions(granted: () -> Unit) {
        permissionHelper.checkOrRequestPermission {
            permissionHelper.checkDozeModePermission()
            granted.invoke()
        }
    }

    data class Tab(
        @DrawableRes val iconRes: Int,
        @DrawableRes val selectedIconRes: Int,
        val text: String,
        val selected: Boolean,
        val screen: @Composable () -> Unit,
    )
}