/*
 * Copyright (c) 2024-2025 Fabian Freund.
 *
 * This file is part of WebLibre
 * (see https://weblibre.eu).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
import 'dart:async';

import 'package:flutter/material.dart';
import 'package:flutter/scheduler.dart';
import 'package:flutter_hooks/flutter_hooks.dart';
import 'package:flutter_mozilla_components/flutter_mozilla_components.dart';
import 'package:go_router/go_router.dart';
import 'package:hooks_riverpod/hooks_riverpod.dart';
import 'package:nullability/nullability.dart';
import 'package:quick_actions/quick_actions.dart';
import 'package:weblibre/core/logger.dart';
import 'package:weblibre/core/providers/router.dart';
import 'package:weblibre/core/routing/routes.dart';
import 'package:weblibre/features/bangs/domain/providers/bangs.dart';
import 'package:weblibre/features/geckoview/domain/providers.dart';
import 'package:weblibre/features/geckoview/domain/providers/browser_extension.dart';
import 'package:weblibre/features/geckoview/domain/providers/selected_tab.dart';
import 'package:weblibre/features/geckoview/domain/providers/tab_session.dart';
import 'package:weblibre/features/geckoview/domain/providers/tab_state.dart';
import 'package:weblibre/features/geckoview/domain/providers/web_extensions_state.dart';
import 'package:weblibre/features/geckoview/domain/repositories/tab.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/providers.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/providers/intent.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/providers/lifecycle.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/services/browser_data.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/services/engine_settings_replication.dart';
import 'package:weblibre/features/geckoview/features/browser/domain/services/proxy_settings_replication.dart';
import 'package:weblibre/features/geckoview/features/history/domain/repositories/history.dart';
import 'package:weblibre/features/geckoview/features/preferences/data/repositories/preference_observer.dart';
import 'package:weblibre/features/geckoview/features/tabs/domain/repositories/container.dart';
import 'package:weblibre/features/share_intent/domain/entities/shared_content.dart';
import 'package:weblibre/features/user/data/models/general_settings.dart';
import 'package:weblibre/features/user/domain/repositories/cache.dart';
import 'package:weblibre/features/user/domain/repositories/general_settings.dart';
import 'package:weblibre/features/user/domain/services/local_authentication.dart';
import 'package:weblibre/features/web_feed/domain/providers/add_dialog_blocking.dart';
import 'package:weblibre/features/web_feed/domain/services/article_content_processor.dart';
import 'package:weblibre/presentation/hooks/on_initialization.dart';
import 'package:weblibre/utils/ui_helper.dart';

class BrowserView extends StatefulHookConsumerWidget {
  final Duration screenshotPeriod;
  final Duration suggestionTimeout;
  final Future<void> Function()? postInitializationStep;
  final StreamSink<Offset>? pointerMoveEventSink;

  const BrowserView({
    this.screenshotPeriod = const Duration(seconds: 10),
    this.suggestionTimeout = const Duration(seconds: 30),
    this.postInitializationStep,
    this.pointerMoveEventSink,
  });

  @override
  ConsumerState<ConsumerStatefulWidget> createState() => _BrowserViewState();
}

class _BrowserViewState extends ConsumerState<BrowserView>
    with WidgetsBindingObserver {
  Timer? _periodicScreenshotUpdate;

  //This is managed by widget state changes to resume a timer
  bool _timerPaused = false;

  DateTime? _suggestionCountTime;

  Future<void> _timerTick(Timer timer) async {
    await ref
        .read(selectedTabSessionProvider)
        .requestScreenshot(requireImageResult: false)
        .onError((error, stackTrace) {
          logger.e(error, stackTrace: stackTrace);
          timer.cancel();

          return null;
        });
  }

  @override
  Widget build(BuildContext context) {
    final initializationCompleter = useMemoized(() => Completer());

    useOnInitialization(() async {
      await ref
          .read(generalSettingsRepositoryProvider.notifier)
          .fetchSettings()
          .then((settings) async {
            await ref
                .read(browserDataServiceProvider.notifier)
                .deleteDataOnEngineStart(settings.deleteBrowsingDataOnQuit);

            if (settings.historyAutoCleanInterval > Duration.zero) {
              await ref
                  .read(historyRepositoryProvider.notifier)
                  .deleteVisitsBetween(
                    DateTime(0),
                    DateTime.now().subtract(settings.historyAutoCleanInterval),
                  );
            }
          });

      // Clear data for containers with clearDataOnExit enabled
      final containersToClear = await ref
          .read(containerRepositoryProvider.notifier)
          .getContainersToClearOnExit();

      if (containersToClear.isNotEmpty) {
        await ref
            .read(browserDataServiceProvider.notifier)
            .clearContainerDataOnEngineStart(containersToClear);
      }
    });

    final hasTab = ref.watch(
      selectedTabProvider.select((value) => value != null),
    );

    ref.listen(
      selectedTabStateProvider.select(
        (state) => (tabId: state?.id, isLoading: state?.isLoading),
      ),
      (previous, next) {
        if (previous?.tabId != next.tabId || next.isLoading == true) {
          _periodicScreenshotUpdate?.cancel();
          _periodicScreenshotUpdate = null;
          _timerPaused = false;
        }

        if (next.isLoading == false &&
            (_periodicScreenshotUpdate?.isActive ?? false) == false) {
          _timerPaused = false;
          _periodicScreenshotUpdate = Timer.periodic(
            widget.screenshotPeriod,
            _timerTick,
          );
        }
      },
    );

    ref.listen(feedRequestedProvider, (previous, next) async {
      if (next.value.mapNotNull(Uri.tryParse) case final Uri url) {
        if (GoRouterState.of(context).topRoute?.name != FeedAddRoute.name) {
          if (ref.read(addFeedDialogBlockingProvider.notifier).canPush(url)) {
            await FeedAddRoute(uri: url.toString()).push(context);
          }
        }
      }
    });

    ref.listen(
      engineBoundIntentStreamProvider,
      (previous, next) async {
        await initializationCompleter.future;

        next.whenData((sharedContent) async {
          final router = await ref.read(routerProvider.future);
          final settings = ref.read(generalSettingsWithDefaultsProvider);

          switch (settings.tabIntentOpenSetting) {
            case TabIntentOpenSetting.regular:
            case TabIntentOpenSetting.private:
              switch (sharedContent) {
                case SharedUrl():
                  await ref
                      .read(tabRepositoryProvider.notifier)
                      .addTab(
                        url: sharedContent.url,
                        private:
                            settings.tabIntentOpenSetting ==
                            TabIntentOpenSetting.private,
                        launchedFromIntent: true,
                        selectTab: true,
                      );
                case SharedText():
                  final bang =
                      ref.read(selectedBangDataProvider()) ??
                      await ref.read(defaultSearchBangDataProvider.future);

                  await ref
                      .read(tabRepositoryProvider.notifier)
                      .addTab(
                        url: bang?.getTemplateUrl(sharedContent.text),
                        private:
                            settings.tabIntentOpenSetting ==
                            TabIntentOpenSetting.private,
                        launchedFromIntent: true,
                        selectTab: true,
                      );
              }
            case TabIntentOpenSetting.ask:
              switch (sharedContent) {
                case SharedUrl():
                  final route = OpenSharedContentRoute(
                    sharedUrl: sharedContent.url.toString(),
                  );
                  await router.push(route.location);
                case SharedText():
                  final route = SearchRoute(
                    tabType:
                        ref.read(selectedTabTypeProvider) ??
                        settings.defaultCreateTabType,
                    searchText: sharedContent.text,
                    launchedFromIntent: true, //launched from intent
                  );
                  await router.push(route.location);
              }
          }
        });
      },
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to engineBoundIntentStreamProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    return Listener(
      behavior: HitTestBehavior.translucent,
      onPointerMove: (widget.pointerMoveEventSink != null)
          ? (event) {
              if (event.down) {
                widget.pointerMoveEventSink!.add(event.localDelta);
              }
            }
          : null,
      child: Stack(
        children: [
          GeckoView(
            preInitializationStep: () async {
              await ref
                  .read(eventServiceProvider)
                  .viewReadyStateEvents
                  .firstWhere((state) => state == true)
                  .timeout(
                    const Duration(seconds: 3),
                    onTimeout: () {
                      logger.e(
                        'Browser fragement not reported ready, trying to intitialize anyways',
                      );
                      return true;
                    },
                  );
            },
            postInitializationStep: () async {
              await widget.postInitializationStep?.call();

              if (!initializationCompleter.isCompleted) {
                const quickActions = QuickActions();

                //Debounce: https://github.com/flutter/flutter/issues/131121
                DateTime? lastAction;
                await quickActions.initialize((type) async {
                  if (lastAction == null ||
                      DateTime.now().difference(lastAction!) >
                          const Duration(seconds: 5)) {
                    if (type == 'new_tab') {
                      lastAction = DateTime.now();

                      final router = await ref.read(routerProvider.future);
                      const route = SearchRoute(tabType: TabType.regular);

                      await router.push(route.location);
                    } else if (type == 'new_private_tab') {
                      lastAction = DateTime.now();

                      final router = await ref.read(routerProvider.future);
                      const route = SearchRoute(tabType: TabType.private);

                      await router.push(route.location);
                    } else {
                      throw UnimplementedError();
                    }
                  }
                });

                await quickActions.setShortcutItems([
                  //TODO: add icons
                  const ShortcutItem(
                    type: 'new_tab',
                    localizedTitle: 'New Tab',
                  ),
                  const ShortcutItem(
                    type: 'new_private_tab',
                    localizedTitle: 'New Private Tab',
                  ),
                ]);

                initializationCompleter.complete();
              }
            },
          ),
          if (!hasTab)
            Positioned.fill(
              child: SizedBox.expand(child: Container(color: Colors.grey[800])),
            ),
        ],
      ),
    );
  }

  @override
  void initState() {
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((_) {
      ref
          .read(browserViewLifecycleProvider.notifier)
          .update(SchedulerBinding.instance.lifecycleState);
    });

    WidgetsBinding.instance.addObserver(this);

    //Initialize and register dependencies
    ref.listenManual(
      fireImmediately: true,
      tabRepositoryProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to tabRepositoryProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      selectionActionServiceProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to selectionActionServiceProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      webExtensionsStateProvider(WebExtensionActionType.browser),
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to webExtensionsStateProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      webExtensionsStateProvider(WebExtensionActionType.page),
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to webExtensionsStateProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      cacheRepositoryProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to cacheRepositoryProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      engineSettingsReplicationServiceProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to engineSettingsReplicationServiceProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      preferenceFixatorProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to preferenceFixatorProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      proxySettingsReplicationProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to proxySettingsReplicationProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );

    ref.listenManual(
      fireImmediately: true,
      articleContentProcessorServiceProvider,
      (previous, next) {},
      onError: (error, stackTrace) {
        logger.e(
          'Error listening to articleContentProcessorServiceProvider',
          error: error,
          stackTrace: stackTrace,
        );
      },
    );
  }

  @override
  void didChangeAppLifecycleState(AppLifecycleState state) {
    super.didChangeAppLifecycleState(state);

    ref.read(browserViewLifecycleProvider.notifier).update(state);

    switch (state) {
      case AppLifecycleState.detached:
      case AppLifecycleState.inactive:
      case AppLifecycleState.hidden:
      case AppLifecycleState.paused:
        if (_periodicScreenshotUpdate?.isActive == true) {
          _periodicScreenshotUpdate?.cancel();
          _timerPaused = true;
        }

        ref
            .read(localAuthenticationServiceProvider.notifier)
            .evictCacheOnBackground();

        if (state == AppLifecycleState.paused) {
          _suggestionCountTime = DateTime.now();
        }
      case AppLifecycleState.resumed:
        if (_timerPaused) {
          _periodicScreenshotUpdate = Timer.periodic(
            widget.screenshotPeriod,
            _timerTick,
          );
          _timerPaused = false;
        }

        if (_suggestionCountTime != null &&
            DateTime.now().difference(_suggestionCountTime!) >
                widget.suggestionTimeout) {
          //Don't do anything if a child route is active
          if (GoRouterState.of(context).topRoute?.name == BrowserRoute.name) {
            final settings = ref.read(generalSettingsWithDefaultsProvider);

            unawaited(
              showSuggestNewTabMessage(
                context,
                onAdd: (searchText) async {
                  await SearchRoute(
                    tabType:
                        ref.read(selectedTabTypeProvider) ??
                        settings.defaultCreateTabType,
                    searchText: searchText ?? SearchRoute.emptySearchText,
                  ).push(context);
                },
              ),
            );
          }
        }

        _suggestionCountTime = null;
    }
  }

  @override
  Future<void> dispose() async {
    WidgetsBinding.instance.removeObserver(this);

    _periodicScreenshotUpdate?.cancel();

    super.dispose();
  }
}
