// Copyright (c) 2020, the Dart project authors. Please see the AUTHORS file
// for details. All rights reserved. Use of this source code is governed by a
// BSD-style license that can be found in the LICENSE file.

import 'package:ffigen/src/code_generator/imports.dart';
import 'package:ffigen/src/config_provider/config.dart';
import 'package:ffigen/src/config_provider/config_types.dart';
import 'package:ffigen/src/header_parser.dart';
import 'package:logging/logging.dart';
import 'package:path/path.dart' as path;
import 'package:test/test.dart';

import '../test_utils.dart';

void main() {
  group('large_test', () {
    setUpAll(() {
      logWarnings(Level.SEVERE);
    });
    test('Libclang test', () {
      final includeDir = path.join(
        packagePathForTests,
        'third_party',
        'libclang',
        'include',
      );
      final logArr = <String>[];
      logToArray(logArr, Level.SEVERE);
      final generator = FfiGenerator(
        output: Output(
          dartFile: Uri.file('unused'),
          commentType: const CommentType(
            CommentStyle.doxygen,
            CommentLength.brief,
          ),
          style: const DynamicLibraryBindings(
            wrapperName: 'LibClang',
            wrapperDocComment: 'Bindings to LibClang.',
          ),
        ),
        headers: Headers(
          compilerOptions: [
            ...defaultCompilerOpts(Logger.root),
            '-I$includeDir',
          ],
          entryPoints: [
            Uri.file(
              path.join(
                packagePathForTests,
                'third_party',
                'libclang',
                'include',
                'clang-c',
                'Index.h',
              ),
            ),
          ],
          include: (Uri header) => [
            'BuildSystem.h',
            'CXCompilationDatabase.h',
            'CXErrorCode.h',
            'CXString.h',
            'Documentation.h',
            'FataErrorHandler.h',
            'Index.h',
          ].any((filename) => header.pathSegments.last == filename),
        ),
        functions: Functions.includeAll,
        structs: Structs.includeAll,
        enums: Enums.includeAll,
        macros: Macros.includeAll,
        typedefs: Typedefs(
          include: (_) => true,
          // ignore: deprecated_member_use_from_same_package
          imported: [ImportedType(ffiImport, 'Int64', 'int', 'time_t')],
        ),
      );
      final library = parse(testContext(generator));

      matchLibraryWithExpected(
        library,
        'large_test_libclang.dart',
        ['test', 'large_integration_tests', '_expected_libclang_bindings.dart'],
        // Remove comments containing @ to hack around a mismatch in the
        // documentation generated by different clang versions.
        codeNormalizer: (code) =>
            code.replaceAll(RegExp('[^\n]*///[^\n]*@[^\n]*\n'), ''),
      );

      const expectedEnumWarnings = [
        'CXAvailabilityKind',
        'CXCallingConv',
        'CXChildVisitResult',
        'CXCompletionChunkKind',
        'CXCursorKind',
        'CXDiagnosticSeverity',
        'CXErrorCode',
        'CXEvalResultKind',
        'CXIdxAttrKind',
        'CXIdxEntityCXXTemplateKind',
        'CXIdxEntityKind',
        'CXIdxEntityLanguage',
        'CXIdxEntityRefKind',
        'CXIdxObjCContainerKind',
        'CXLanguageKind',
        'CXLinkageKind',
        'CXLoadDiag_Error',
        'CXPrintingPolicyProperty',
        'CXRefQualifierKind',
        'CXResult',
        'CXSymbolRole',
        'CXTLSKind',
        'CXTUResourceUsageKind',
        'CXTemplateArgumentKind',
        'CXTokenKind',
        'CXTypeKind',
        'CXTypeNullabilityKind',
        'CXVisibilityKind',
        'CXVisitorResult',
        'CX_CXXAccessSpecifier',
        'CX_StorageClass',
      ];
      final actualLogs = logArr.join('\n');
      for (final e in expectedEnumWarnings) {
        expect(actualLogs, contains(e));
      }
    });

    test('CJSON test', () {
      final generator = FfiGenerator(
        output: Output(
          dartFile: Uri.file('unused'),
          style: const DynamicLibraryBindings(
            wrapperName: 'CJson',
            wrapperDocComment: 'Bindings to Cjson.',
          ),
        ),
        headers: Headers(
          entryPoints: [
            Uri.file(
              path.join(
                packagePathForTests,
                'third_party',
                'cjson_library',
                'cJSON.h',
              ),
            ),
          ],
          include: (Uri header) => header.pathSegments.last == 'cJSON.h',
        ),
        functions: Functions.includeAll,
        structs: Structs.includeAll,
        macros: Macros.includeAll,
        typedefs: Typedefs.includeAll,
      );
      final library = parse(testContext(generator));

      matchLibraryWithExpected(library, 'large_test_cjson.dart', [
        'test',
        'large_integration_tests',
        '_expected_cjson_bindings.dart',
      ]);
    });

    test('SQLite test', () {
      // Excluding functions that use 'va_list' because it can either be a
      // Pointer<__va_list_tag> or int depending on the OS.
      final generator = FfiGenerator(
        output: Output(
          dartFile: Uri.file('unused'),
          style: const DynamicLibraryBindings(
            wrapperName: 'SQLite',
            wrapperDocComment: 'Bindings to SQLite.',
          ),
          commentType: const CommentType(CommentStyle.any, CommentLength.full),
        ),
        headers: Headers(
          entryPoints: [
            Uri.file(
              path.join(
                packagePathForTests,
                'third_party',
                'sqlite',
                'sqlite3.h',
              ),
            ),
          ],
          include: (Uri header) => header.pathSegments.last == 'sqlite3.h',
        ),
        functions: Functions(
          include: (declaration) => !{
            'sqlite3_vmprintf',
            'sqlite3_vsnprintf',
            'sqlite3_str_vappendf',
          }.contains(declaration.originalName),
        ),
        structs: Structs.includeAll,
        globals: Globals.includeAll,
        macros: Macros.includeAll,
        typedefs: Typedefs.includeAll,
      );
      final library = parse(testContext(generator));

      matchLibraryWithExpected(library, 'large_test_sqlite.dart', [
        'test',
        'large_integration_tests',
        '_expected_sqlite_bindings.dart',
      ]);
    });
  });
}
