package eu.lepiller.nani;

import android.app.PendingIntent;
import android.app.Service;
import android.content.Intent;
import android.os.Handler;
import android.os.IBinder;
import android.os.Looper;
import android.util.Log;
import android.util.Pair;

import androidx.annotation.Nullable;
import androidx.core.app.NotificationCompat;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import eu.lepiller.nani.dictionary.Dictionary;
import eu.lepiller.nani.dictionary.DictionaryFactory;

public class DictionaryDownloadService extends Service {
    private NotificationCompat.Builder builder;
    private DownloadQueue downloadQueue;

    static public class DownloadData {
        public String currentName;
        public int currentProgress;
        public ArrayList<String> downloading;

        DownloadData(String name, int progress, ArrayList<String> next) {
            currentName = name;
            currentProgress = progress;
            downloading = next;
        }
    }

    private static MutableLiveData<DownloadData> data;

    public static final String DOWNLOAD_ACTION = "eu.lepiller.nani.action.DOWNLOAD";
    public static final String PAUSE_ACTION = "eu.lepiller.nani.action.PAUSE";
    private static final String TAG = "DicoDownloadService";

    public static LiveData<DownloadData> getData() {
        if(data == null)
            data = new MutableLiveData<>();

        return data;
    }

    @Override
    public void onCreate() {
        Log.d(TAG, "onCreate");
        super.onCreate();
        builder =  new NotificationCompat.Builder(this, App.DICTIONARY_DOWNLOAD_NOTIFICATION_CHANNEL)
                .setSmallIcon(R.drawable.ic_launcher_foreground)
                .setContentText(getString(R.string.downloading))
                .setOnlyAlertOnce(true);
        downloadQueue = new DownloadQueue();
        Downloader downloadThread = new Downloader(() -> {
            stopForeground(true);
            stopSelf();
        });
        new Thread(downloadThread).start();
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        String action = intent.getAction();
        Log.d(TAG, "onStartCommand: " + action);
        if(action == null) {
            return START_NOT_STICKY;
        } else if(action.equals(DOWNLOAD_ACTION)) {
            String name = intent.getStringExtra(DictionaryDownloadActivity.EXTRA_DICTIONARY);
            downloadQueue.addJob(name);
            updateNotification(-1, downloadQueue.nextJob());
        } else if(action.equals(PAUSE_ACTION)) {
            downloadQueue.stop();
        }

        return START_NOT_STICKY;
    }

    @Override
    public void onDestroy() {
        Log.d(TAG, "onDestroy");
        downloadQueue.clear();
        updateNotification(0, null);
        downloadQueue = null;
        super.onDestroy();
    }

    void updateNotification(int progress, String name) {
        PendingIntent pendingIntent;
        if(name == null) {
            Intent notificationIntent = new Intent(this, DictionaryActivity.class);
            pendingIntent = PendingIntent.getActivity(this, 0, notificationIntent, 0);
            builder.setContentTitle(getString(R.string.downloading));
        } else {
            Intent notificationIntent = new Intent(this, DictionaryDownloadActivity.class);
            notificationIntent.putExtra(DictionaryDownloadActivity.EXTRA_DICTIONARY, name);
            pendingIntent = PendingIntent.getActivity(this, 0, notificationIntent, 0);
            builder.setContentTitle(name);
        }

        builder.setContentIntent(pendingIntent)
               .setProgress(100, progress, (progress < 0));
        startForeground(1, builder.build());

        final DownloadData downloadData = new DownloadData(name, progress, downloadQueue.downloadQueue());

        Handler threadHandler = new Handler(Looper.getMainLooper());
        threadHandler.post(() -> {
            if(data == null)
                data = new MutableLiveData<>();
            data.setValue(downloadData);
        });
    }

    private static class DownloadQueue {
        private final ArrayList<String> downloadQueue = new ArrayList<>();
        private boolean waitsForFirstWork = true;
        private boolean stopAsked = false;
        private final ReentrantLock mutex = new ReentrantLock();

        ArrayList<String> downloadQueue() {
            try {
                mutex.lock();
                return new ArrayList<>(downloadQueue);
            } finally {
                mutex.unlock();
            }
        }

        void clear() {
            try {
                mutex.lock();
                downloadQueue.clear();
            } finally {
                mutex.unlock();
            }
        }

        String popNextJob() {
            try {
                mutex.lock();
                if(downloadQueue.size() > 0) {
                    String name = downloadQueue.get(0);
                    downloadQueue.remove(0);
                    return name;
                }
                return null;
            } finally {
                mutex.unlock();
            }
        }

        String nextJob() {
            try {
                mutex.lock();
                if(downloadQueue.size() > 0)
                    return downloadQueue.get(0);
                return null;
            } finally {
                mutex.unlock();
            }
        }

        void addJob(String name) {
            try {
                mutex.lock();
                waitsForFirstWork = false;
                downloadQueue.add(name);
            } finally {
                mutex.unlock();
            }
        }

        boolean isWaitingForFirstJob() {
            try {
                mutex.lock();
                return waitsForFirstWork;
            } finally {
                mutex.unlock();
            }
        }

        void stop() {
            try {
                mutex.lock();
                stopAsked = true;
                downloadQueue.clear();
            } finally {
                mutex.unlock();
            }
        }

        boolean wantsStop() {
            try {
                mutex.lock();
                return stopAsked;
            } finally {
                mutex.unlock();
            }
        }
    }

    private class Downloader implements Runnable {
        private final Runnable onStop;

        Downloader(Runnable onStop) {
            this.onStop = onStop;
        }

        @Override
        public void run() {
            while(downloadQueue.isWaitingForFirstJob()) {
                try {
                    Thread.sleep(100);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            String name;
            while((name = downloadQueue.popNextJob()) != null) {
                updateNotification(-1, null);
                doDownload(name);
                updateNotification(-1, null);
            }

            Handler threadHandler = new Handler(Looper.getMainLooper());
            threadHandler.post(onStop);
        }

        private void publishProgress(int progress, String name) {
            updateNotification(progress, name);
        }

        private void doDownload(String name) {
            Dictionary d = DictionaryFactory.getByName(DictionaryDownloadService.this, name);
            if(d == null)
                return;

            for (Map.Entry<String, Pair<File, File>> e : d.getDownloads().entrySet()) {
                try {
                    String uri = e.getKey();
                    File temporaryFile = e.getValue().first;
                    File cacheFile = e.getValue().second;

                    boolean newFile = downloadSha256(new URL(uri + ".sha256"), new File(cacheFile + ".sha256"));
                    if(newFile) {
                        d.remove();
                    }

                    long expectedFileLength = getRange(new URL(uri));
                    downloadFile(new URL(uri), temporaryFile, expectedFileLength, name);
                } catch (Exception exception) {
                    exception.printStackTrace();
                }
            }

            d.switchToCacheFile();
        }

        private boolean downloadSha256(URL url, File dest) throws IOException {
            createParent(dest);
            byte[] data = new byte[4096];
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.connect();
            if(connection.getResponseCode() != HttpURLConnection.HTTP_OK)
                return true;

            InputStream input = connection.getInputStream();
            File old_file = new File(dest + ".old");
            deleteIfExists(old_file);
            if(dest.exists())
                rename(dest, old_file);

            FileOutputStream output = new FileOutputStream(dest);

            int count;
            while((count = input.read(data)) != -1) {
                if (isCancelled()) {
                    input.close();
                    return false;
                }
                output.write(data, 0, count);
            }

            if(old_file.exists()) {
                // Check that we continue to download the same file
                String old_hash = Dictionary.readSha256FromFile(old_file);
                String new_hash = Dictionary.readSha256FromFile(dest);
                return old_hash.compareTo(new_hash) != 0;
            }
            return true;
        }

        private long getRange(URL url) throws IOException {
            long expectedLength;
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("HEAD");
            connection.connect();
            if(connection.getResponseCode() != HttpURLConnection.HTTP_OK)
                return -1;

            boolean acceptRanges = connection.getHeaderFields().containsKey("accept-ranges");
            expectedLength = connection.getContentLength();
            List<String> headers = connection.getHeaderFields().get("accept-ranges");
            if(headers != null) {
                for (String h : headers) {
                    if (h.toLowerCase().compareTo("none") == 0)
                        acceptRanges = false;
                }
            }

            if(acceptRanges)
                return expectedLength;
            return -1;
        }

        private void downloadFile(URL url, File dest, long expectedLength, String name) throws IOException {
            createParent(dest);
            long total = 0;
            byte[] data = new byte[4096];
            FileOutputStream output;

            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            if(expectedLength > 0 && dest.length() < expectedLength) {
                connection.addRequestProperty("Range", "bytes=" + dest.length() + "-" + (expectedLength-1));
                total = dest.length();
                output = new FileOutputStream(dest, true);
            } else {
                output = new FileOutputStream(dest);
            }
            connection.connect();

            // expect HTTP 200 OK, so we don't mistakenly save error report
            // instead of the file
            if (connection.getResponseCode() != HttpURLConnection.HTTP_OK &&
                    connection.getResponseCode() != HttpURLConnection.HTTP_PARTIAL) {
                Log.e(TAG, "Server returned HTTP " + connection.getResponseCode()
                        + " " + connection.getResponseMessage());
                return;
            }

            int fileLength = connection.getContentLength();
            InputStream input = connection.getInputStream();

            int count;
            int lastNotifiedProgress = 0;
            publishProgress(0, name);
            while ((count = input.read(data)) != -1) {
                // allow canceling with back button
                if (isCancelled()) {
                    input.close();
                    return;
                }
                total += count;
                output.write(data, 0, count);
                // publishing the progress....
                if (fileLength > 0) {// only if total length is known
                    float progress = (int) (total * 100 / fileLength);
                    if(lastNotifiedProgress < progress - 2) {
                        lastNotifiedProgress = (int)progress;
                        publishProgress((int) progress, name);
                        output.flush();
                    }
                }
            }
        }

        private boolean isCancelled() {
            return downloadQueue.wantsStop();
        }

        private void createParent(File file) {
            if(file.getParentFile() == null || (!file.getParentFile().exists() && !file.getParentFile().mkdirs()))
                Log.w(TAG, "could not create parent of " + file);
        }

        private void deleteIfExists(File file) {
            if(file.exists()) {
                if(!file.delete())
                    Log.w(TAG, "could not delete file " + file);
            }
        }

        private void rename(File file, File dest) {
            if(file.exists()) {
                if(!file.renameTo(dest))
                    Log.w(TAG, "could not rename "+ file + " to " + dest);
            }
        }
    }

    @Nullable
    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }
}
