package eu.darken.sdmse.appcleaner.core.forensics.filter

import dagger.Binds
import dagger.Module
import dagger.Reusable
import dagger.hilt.InstallIn
import dagger.hilt.components.SingletonComponent
import dagger.multibindings.IntoSet
import eu.darken.sdmse.appcleaner.core.AppCleanerSettings
import eu.darken.sdmse.appcleaner.core.forensics.BaseExpendablesFilter
import eu.darken.sdmse.appcleaner.core.forensics.ExpendablesFilter
import eu.darken.sdmse.appcleaner.core.forensics.sieves.JsonAppSieve
import eu.darken.sdmse.common.areas.DataArea
import eu.darken.sdmse.common.datastore.value
import eu.darken.sdmse.common.debug.logging.log
import eu.darken.sdmse.common.debug.logging.logTag
import eu.darken.sdmse.common.files.APath
import eu.darken.sdmse.common.files.APathLookup
import eu.darken.sdmse.common.files.GatewaySwitch
import eu.darken.sdmse.common.files.Segments
import eu.darken.sdmse.common.files.lowercase
import eu.darken.sdmse.common.pkgs.Pkg
import eu.darken.sdmse.common.storage.StorageEnvironment
import javax.inject.Inject
import javax.inject.Provider


@Reusable
class ThumbnailsFilter @Inject constructor(
    private val jsonBasedSieveFactory: JsonAppSieve.Factory,
    environment: StorageEnvironment,
    private val gatewaySwitch: GatewaySwitch,
) : BaseExpendablesFilter() {

    private val cacheFolderPrefixes = environment.ourCacheDirs.map { it.name }

    private lateinit var sieve: JsonAppSieve

    override suspend fun initialize() {
        log(TAG) { "initialize()" }
        sieve = jsonBasedSieveFactory.create("expendables/db_thumbnail_files.json")
    }

    override suspend fun match(
        pkgId: Pkg.Id,
        target: APathLookup<APath>,
        areaType: DataArea.Type,
        pfpSegs: Segments
    ): ExpendablesFilter.Match? {
        val lcsegments = pfpSegs.lowercase()

        if (lcsegments.isNotEmpty() && IGNORED_FILES.contains(lcsegments[lcsegments.size - 1])) {
            return null
        }

        if (lcsegments.size >= 2 && HIDDEN_FOLDERS.contains(lcsegments[0])) {
            return target.toDeletionMatch()
        }

        // Default case, we don't handle that.
        // package/cache/file
        if (lcsegments.size >= 2 && pkgId.name == lcsegments[0] && cacheFolderPrefixes.contains(lcsegments[1])) {
            return null
        }


        // package/thumbs.dat
        if (lcsegments.size == 2 && HIDDEN_FILES.contains(lcsegments[1])) {
            return target.toDeletionMatch()
        }

        // package/files/thumbs.dat
        if (lcsegments.size == 3 && HIDDEN_FILES.contains(lcsegments[2])) {
            return target.toDeletionMatch()
        }

        //    0      1     2
        // package/.thumbnails/file
        if (lcsegments.size >= 3 && HIDDEN_FOLDERS.contains(lcsegments[1])) {
            return target.toDeletionMatch()
        }

        //    0      1     2     3
        // package/files/.thumbnails/file
        if (lcsegments.size >= 4 && "files" == lcsegments[1] && HIDDEN_FOLDERS.contains(lcsegments[2])) {
            return target.toDeletionMatch()
        }

        //    -1     0      1     2     3
        // sdcard/Huawei/Themes/.cache/file
        if (lcsegments.size >= 4 && areaType == DataArea.Type.SDCARD && HIDDEN_FOLDERS.contains(lcsegments[2])) {
            return target.toDeletionMatch()
        }

        return if (pfpSegs.isNotEmpty() && sieve.matches(pkgId, areaType, pfpSegs)) {
            target.toDeletionMatch()
        } else {
            null
        }
    }

    override suspend fun process(
        targets: Collection<ExpendablesFilter.Match>,
        allMatches: Collection<ExpendablesFilter.Match>
    ): ExpendablesFilter.ProcessResult {
        return deleteAll(
            targets.map { it as ExpendablesFilter.Match.Deletion },
            gatewaySwitch,
            allMatches
        )
    }

    @Reusable
    class Factory @Inject constructor(
        private val settings: AppCleanerSettings,
        private val filterProvider: Provider<ThumbnailsFilter>
    ) : ExpendablesFilter.Factory {
        override suspend fun isEnabled(): Boolean = settings.filterThumbnailsEnabled.value()
        override suspend fun create(): ExpendablesFilter = filterProvider.get()
    }

    @InstallIn(SingletonComponent::class)
    @Module
    abstract class DIM {
        @Binds @IntoSet abstract fun mod(mod: Factory): ExpendablesFilter.Factory
    }

    companion object {
        private val HIDDEN_FOLDERS: Collection<String> = listOf(
            ".thumbs",
            "thumbs",
            ".thumbnails",
            "thumbnails",
            "albumthumbs",
        )
        private val HIDDEN_FILES: Collection<String> = listOf(

        )
        private val IGNORED_FILES: Collection<String> = listOf(
            ".nomedia"
        )
        private val TAG = logTag("AppCleaner", "Scanner", "Filter", "Thumbnails")
    }
}