package eu.darken.capod.main.ui.overview

import android.app.Activity
import androidx.lifecycle.LiveData
import androidx.lifecycle.SavedStateHandle
import dagger.hilt.android.lifecycle.HiltViewModel
import eu.darken.capod.common.bluetooth.BluetoothManager2
import eu.darken.capod.common.coroutine.DispatcherProvider
import eu.darken.capod.common.debug.DebugSettings
import eu.darken.capod.common.debug.logging.log
import eu.darken.capod.common.debug.logging.logTag
import eu.darken.capod.common.flow.combine
import eu.darken.capod.common.flow.throttleLatest
import eu.darken.capod.common.livedata.SingleLiveEvent
import eu.darken.capod.common.permissions.Permission
import eu.darken.capod.common.uix.ViewModel3
import eu.darken.capod.common.upgrade.UpgradeRepo
import eu.darken.capod.main.core.GeneralSettings
import eu.darken.capod.main.core.MonitorMode
import eu.darken.capod.main.core.PermissionTool
import eu.darken.capod.main.ui.overview.cards.BluetoothDisabledVH
import eu.darken.capod.main.ui.overview.cards.MonitoringActiveVH
import eu.darken.capod.main.ui.overview.cards.NoProfilesVH
import eu.darken.capod.main.ui.overview.cards.PermissionCardVH
import eu.darken.capod.main.ui.overview.cards.UnmatchedDevicesCardVH
import eu.darken.capod.main.ui.overview.cards.pods.DualPodsCardVH
import eu.darken.capod.main.ui.overview.cards.pods.SinglePodsCardVH
import eu.darken.capod.main.ui.overview.cards.pods.UnknownPodDeviceCardVH
import eu.darken.capod.monitor.core.PodMonitor
import eu.darken.capod.monitor.core.worker.MonitorControl
import eu.darken.capod.pods.core.DualPodDevice
import eu.darken.capod.pods.core.PodDevice
import eu.darken.capod.pods.core.SinglePodDevice
import eu.darken.capod.profiles.core.DeviceProfilesRepo
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.channelFlow
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.flatMapLatest
import kotlinx.coroutines.flow.flowOf
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.isActive
import java.time.Instant
import javax.inject.Inject

@HiltViewModel
class OverviewFragmentVM @Inject constructor(
    @Suppress("UNUSED_PARAMETER") handle: SavedStateHandle,
    dispatcherProvider: DispatcherProvider,
    private val monitorControl: MonitorControl,
    private val podMonitor: PodMonitor,
    private val permissionTool: PermissionTool,
    private val generalSettings: GeneralSettings,
    debugSettings: DebugSettings,
    private val upgradeRepo: UpgradeRepo,
    private val bluetoothManager: BluetoothManager2,
    private val profilesRepo: DeviceProfilesRepo,
) : ViewModel3(dispatcherProvider = dispatcherProvider) {

    init {
        if (!generalSettings.isOnboardingDone.value) {
            OverviewFragmentDirections.actionOverviewFragmentToOnboardingFragment().navigate()
        }
    }

    val upgradeState = upgradeRepo.upgradeInfo
        .onEach {
            if (!it.isPro && it.error != null) {
                errorEvents.postValue(it.error)
            }
        }
        .asLiveData2()
    val launchUpgradeFlow = SingleLiveEvent<(Activity) -> Unit>()

    private val updateTicker = channelFlow<Unit> {
        while (isActive) {
            trySend(Unit)
            delay(3000)
        }
    }

    val workerAutolaunch: LiveData<Unit> = permissionTool.missingPermissions
        .onEach {
            if (it.isNotEmpty()) {
                log(TAG) { "Missing permissions: $it" }
                return@onEach
            }

            val shouldStartMonitor = when (generalSettings.monitorMode.value) {
                MonitorMode.MANUAL -> false
                MonitorMode.AUTOMATIC -> bluetoothManager.connectedDevices.first().isNotEmpty()
                MonitorMode.ALWAYS -> true
            }
            if (shouldStartMonitor) {
                log(TAG) { "Starting monitor" }
                monitorControl.startMonitor()
            }
        }
        .map { }
        .asLiveData2()

    val requestPermissionEvent = SingleLiveEvent<Permission>()

    private var showUnmatchedDevices = false

    private val pods: Flow<List<PodDevice>> = permissionTool.missingPermissions
        .flatMapLatest { permissions ->
            if (permissions.isNotEmpty()) {
                return@flatMapLatest flowOf(emptyList())
            }

            podMonitor.devices
        }
        .catch { errorEvents.postValue(it) }
        .throttleLatest(1000)

    val listItems: LiveData<List<OverviewAdapter.Item>> = combine(
        updateTicker,
        permissionTool.missingPermissions,
        pods,
        debugSettings.isDebugModeEnabled.flow,
        bluetoothManager.isBluetoothEnabled,
        profilesRepo.profiles,
    ) { _, permissions, devices, isDebugMode, isBluetoothEnabled, profiles ->
        val items = mutableListOf<OverviewAdapter.Item>()

        permissions
            .map { perm ->
                PermissionCardVH.Item(
                    permission = perm,
                    onRequest = { requestPermissionEvent.postValue(it) },
                )
            }
            .run { items.addAll(this) }

        if (permissions.isEmpty()) {
            if (!isBluetoothEnabled) {
                items.add(0, BluetoothDisabledVH.Item)
            } else if (profiles.isEmpty()) {
                items.add(
                    0, NoProfilesVH.Item(
                        onManageDevices = {
                            OverviewFragmentDirections.actionOverviewFragmentToDeviceManagerFragment().navigate()
                        }
                    ))
            }
        }

        if (permissions.isEmpty() && isBluetoothEnabled) {
            val now = Instant.now()
            
            // Split devices into profiled and unmatched
            val profiledDevices = devices.filter { it.meta.profile != null }
            val unmatchedDevices = devices.filter { it.meta.profile == null }
            
            // Add profiled devices first
            profiledDevices.map { device ->
                when (device) {
                    is DualPodDevice -> DualPodsCardVH.Item(
                        now = now,
                        device = device,
                        showDebug = isDebugMode,
                    )

                    is SinglePodDevice -> SinglePodsCardVH.Item(
                        now = now,
                        device = device,
                        showDebug = isDebugMode,
                    )

                    else -> UnknownPodDeviceCardVH.Item(
                        now = now,
                        device = device,
                        showDebug = isDebugMode,
                    )
                }
            }.run { items.addAll(this) }

            if (profiles.isNotEmpty() && devices.isEmpty()) {
                items.add(MonitoringActiveVH.Item)
            }

            // Add unmatched devices section if any exist
            if (unmatchedDevices.isNotEmpty()) {
                items.add(UnmatchedDevicesCardVH.Item(
                    count = unmatchedDevices.size,
                    isExpanded = showUnmatchedDevices,
                    onToggle = { toggleUnmatchedDevices() }
                ))
                
                // Show unmatched devices if expanded
                if (showUnmatchedDevices) {
                    unmatchedDevices.map { device ->
                        when (device) {
                            is DualPodDevice -> DualPodsCardVH.Item(
                                now = now,
                                device = device,
                                showDebug = isDebugMode,
                            )

                            is SinglePodDevice -> SinglePodsCardVH.Item(
                                now = now,
                                device = device,
                                showDebug = isDebugMode,
                            )

                            else -> UnknownPodDeviceCardVH.Item(
                                now = now,
                                device = device,
                                showDebug = isDebugMode,
                            )
                        }
                    }.run { items.addAll(this) }
                }
            }
        }

        items
    }
        .catch { errorEvents.postValue(it) }
        .asLiveData2()

    fun onPermissionResult(granted: Boolean) {
        if (granted) permissionTool.recheck()
    }

    fun goToSettings() = launch {
        OverviewFragmentDirections.actionOverviewFragmentToSettingsFragment().navigate()
    }

    fun goToDeviceManager() = launch {
        OverviewFragmentDirections.actionOverviewFragmentToDeviceManagerFragment().navigate()
    }

    fun onUpgrade() = launch {
        val call: (Activity) -> Unit = {
            upgradeRepo.launchBillingFlow(it)
        }
        launchUpgradeFlow.postValue(call)
    }

    private fun toggleUnmatchedDevices() {
        showUnmatchedDevices = !showUnmatchedDevices
    }

    companion object {
        private val TAG = logTag("Overview", "OverviewFragmentVM")
    }
}