import { useTranslation } from "react-i18next";
import { FlatList } from "react-native";

import { NetworkError } from "../api/RadioBrowser";
import { useStore } from "../store";
import { Splash } from "./Splash";
import { StationListItem } from "./StationListItem";

export function StationList({ data, error, loading, emptyIcon, emptyText, retry, role, style }) {
  const { t } = useTranslation();

  // Must access player fields directly, because player itself updates very often and there are
  // three in-memory station lists that can contain a lot of entries.
  const playerID = useStore((state) => state.player.id);
  const playerPlaying = useStore((state) => state.player.playing);

  if (loading) {
    return (
      <Splash icon="Radio" style={{ ...style }}>
        {t("loading")}
      </Splash>
    );
  }

  if (error != null) {
    let icon, message;

    switch (error.message) {
      case NetworkError.Connection:
        icon = "Unplug";
        message = t("net_error_connection");
        break;
      case NetworkError.Parse:
      case NetworkError.Response:
        icon = "FileDigit";
        message = t("net_error_response");
        break;
      default:
        icon = "OctagonAlert";
        message = t("net_error_other");
    }

    return (
      <Splash action={retry} actionText={t("button_refresh")} icon={icon} style={{ ...style }}>
        {message}
      </Splash>
    );
  }

  if (data.length === 0) {
    return (
      <Splash icon={emptyIcon} style={{ ...style }}>
        {emptyText}
      </Splash>
    );
  }

  return (
    <FlatList
      data={data}
      keyExtractor={(item) => `${item.id}-${item.timestamp}`}
      renderItem={({ item, key }) => (
        <StationListItem
          data={item}
          player={{ id: playerID, playing: playerPlaying }}
          role={role}
          key={key}
        />
      )}
      style={{ ...style }}
    />
  );
}
