import { Star } from "lucide-react-native";
import { useEffect } from "react";
import { useTranslation } from "react-i18next";
import { I18nManager, ToastAndroid, View } from "react-native";
import { IconButton, Surface, Text, useTheme } from "react-native-paper";
import { useSafeAreaInsets } from "react-native-safe-area-context";

import i18next from "../i18n";
import { Status, nextTrack, pauseTrack, playTrack, setupPlayer } from "../services/PlaybackService";
import { useStore } from "../store";
import { styles } from "../styles";

function getPlayerText(state) {
  const t = i18next.t;

  switch (state.status) {
    case Status.None:
    case Status.Paused:
    case Status.Stopped:
    case Status.Ended:
      return t("player_stopped");

    case Status.Loading:
      return t("player_loading");

    case Status.Buffering:
    case Status.Ready:
      return t("player_buffering");

    case Status.Playing:
      if (state.title) {
        return `${state.artist} - ${state.title}`;
      } else {
        return t("player_playing");
      }

    case Status.Error:
      return t("player_error", { error: state.error });

    default:
      return t("player_stopped");
  }
}

export function Player() {
  const rtl = I18nManager.isRTL;

  const safeAreaInsets = useSafeAreaInsets();
  const theme = useTheme();
  const { t } = useTranslation();

  const player = useStore((state) => state.player);
  const bookmarks = useStore((state) => state.bookmarks);

  const isBookmark = bookmarks.map((b) => b.id).includes(player.id);
  const isPausable = [Status.Loading, Status.Buffering, Status.Ready, Status.Playing].includes(
    player.status,
  );

  useEffect(() => {
    setupPlayer();
  }, []);

  return (
    <View
      style={{
        bottom: styles.navbar.height + safeAreaInsets.bottom,
        height: styles.player.height + styles.player.margin,
        paddingBottom: styles.player.margin,
        paddingHorizontal: styles.player.margin,
        position: "absolute",
        width: "100%",
      }}
    >
      <Surface
        elevation={2}
        mode="flat"
        style={{ borderRadius: 20, gap: 4, height: "100%", justifyContent: "center" }}
      >
        <View style={{ paddingHorizontal: 64 }}>
          <Text
            numberOfLines={2}
            style={{ letterSpacing: 0.5, textAlign: "center" }}
            variant="bodyLarge"
          >
            {getPlayerText(player)}
          </Text>
        </View>
        {player.name && (
          <View
            style={{
              alignItems: "center",
              flexDirection: "row",
              gap: 8,
              justifyContent: "center",
              paddingHorizontal: 64,
            }}
          >
            {isBookmark && <Star size={12} style={{ color: theme.colors.primary, marginTop: 2 }} />}
            <Text numberOfLines={1} style={{ color: theme.colors.primary, letterSpacing: 0.5 }}>
              {player.name}
            </Text>
          </View>
        )}
        {player.id && (
          <View style={{ flexDirection: rtl ? "row-reverse" : "row", position: "absolute" }}>
            <IconButton
              contentStyle={{ alignItems: rtl ? "flex-end" : "flex-start", padding: 24 }}
              icon={isPausable ? "Pause" : "Play"}
              iconColor={theme.colors.primary}
              onPress={() => (isPausable ? pauseTrack() : playTrack())}
              style={{ flexGrow: 1, height: styles.player.height, margin: 0 }}
            />
            <IconButton
              contentStyle={{ alignItems: rtl ? "flex-start" : "flex-end", padding: 24 }}
              icon="SkipForward"
              iconColor={theme.colors.primary}
              onPress={async () => {
                if (!(await nextTrack())) {
                  ToastAndroid.show(t("player_bookmarks_empty"), ToastAndroid.SHORT);
                }
              }}
              style={{ flexGrow: 1, height: styles.player.height, margin: 0 }}
            />
          </View>
        )}
      </Surface>
    </View>
  );
}
