import axios from "axios";
import DeviceCountry from "react-native-device-country";

import { API_ROOT, API_USER_AGENT, MAX_ENTRIES, NET_TIMEOUT } from "../const";
import { getLogger } from "../logging";
import { useStore } from "../store";

export const NetworkError = {
  Connection: "connection",
  Parse: "parse",
  Response: "response",
  Other: "other",
};

const client = axios.create({
  headers: { "User-Agent": API_USER_AGENT },
  timeout: NET_TIMEOUT,
  transformResponse: (res) => JSON.parse(res), // explicit, raises ParseError
});

const log = getLogger("RadioBrowser");

let currentServer;

function cleanCodecData(data) {
  const codec = data.codec
    .split(",")
    .map((c) => c.toUpperCase())
    .filter((c) => !["UNKNOWN", "H.264"].includes(c))[0];
  const bitrate = data.bitrate || null;

  if (codec && bitrate) {
    return `${codec} ${bitrate}kb/s`;
  }
  if (codec) {
    return `${codec}`;
  }
  if (bitrate) {
    return `??? ${bitrate}kb/s`;
  }

  return null;
}

function cleanData(data) {
  return {
    id: data.stationuuid,
    name: (data.name || "").trim(),
    url: data.url,
    artwork: data.favicon,
    codec: cleanCodecData(data),
    valid: !!data.lastcheckok,
    timestamp: +new Date(),
  };
}

async function getCountryCode() {
  const customCountry = useStore.getState().customCountry;
  if (customCountry) {
    return customCountry;
  }

  const data = await DeviceCountry.getCountryCode();
  if (data?.code) {
    return data.code;
  }

  return "us";
}

function getCurrentServer(shuffle) {
  if (!currentServer || shuffle) {
    const serverList = [...useStore.getState().serverList];
    const serverIdx = serverList.indexOf(currentServer);
    if (serverList.length > 1 && serverIdx !== -1) {
      serverList.splice(serverIdx, 1);
    }

    currentServer = serverList[+new Date() % serverList.length];
    useStore.setState({ serverCurrent: currentServer });
  }

  return currentServer;
}

function parseError(err) {
  if (axios.isAxiosError(err)) {
    if (err.status) {
      return NetworkError.Response;
    } else {
      return NetworkError.Connection;
    }
  }
  if (err.toString().toLowerCase().includes("parse")) {
    return NetworkError.Parse;
  }

  return NetworkError.Other;
}

export async function getServerList() {
  log.i("get server list");

  return await client
    .get(`${API_ROOT}/json/servers`)
    .then((res) => {
      let data = [...res.data].filter((s) => s?.name).map((s) => `https://${s.name}`);
      data = [...new Set(data)];
      data.sort();
      return data;
    })
    .catch((err) => {
      let reason = parseError(err);
      log.e(`get server list fail reason=${reason} msg=${err}`);
      throw new Error(reason);
    });
}

export async function getStationsLocal(changeServer) {
  const server = getCurrentServer(changeServer);
  log.i(`get local stations server=${server}`);

  return await getCountryCode()
    .then((code) =>
      client.post(`${server}/json/stations/bycountrycodeexact/${code}`, {
        order: "clickcount",
        reverse: true,
        offset: 0,
        limit: MAX_ENTRIES,
      }),
    )
    .then((res) => [...res.data].map(cleanData))
    .catch((err) => {
      let reason = parseError(err);
      log.e(`get local stations server=${server} fail reason=${reason} msg=${err}`);
      throw new Error(reason);
    });
}

export async function getStationsByName(name, changeServer) {
  const server = getCurrentServer(changeServer);
  log.i(`get stations by name server=${server}`);

  return await client
    .post(`${server}/json/stations/byname/${encodeURI(name)}`, {
      order: "clickcount",
      reverse: true,
      offset: 0,
      limit: MAX_ENTRIES,
    })
    .then((res) => [...res.data].map(cleanData))
    .catch((err) => {
      let reason = parseError(err);
      log.e(`get stations by name server=${server} fail reason=${reason} msg=${err}`);
      throw new Error(reason);
    });
}

export async function getStationsByUUIDs(uuids, changeServer) {
  const server = getCurrentServer(changeServer);
  log.i(`get stations by uuid server=${server}`);

  return await client
    .post(`${server}/json/stations/byuuid`, { uuids })
    .then((res) => [...res.data].map(cleanData))
    .catch((err) => {
      let reason = parseError(err);
      log.e(`get stations by uuid server=${server} fail reason=${reason} msg=${err}`);
      throw new Error(reason);
    });
}

export async function sendPlaybackStats(uuid) {
  if (!useStore.getState().stats) {
    return;
  }

  const server = getCurrentServer();
  log.i(`send playback stats server=${server}`);

  return await client
    .get(`${server}/json/url/${encodeURI(uuid)}`)
    .then(() => ({}))
    .catch((err) => {
      let reason = parseError(err);
      log.e(`send playback stats server=${server} fail reason=${reason} msg=${err}`);
      throw new Error(reason);
    });
}
