import { useState } from "react";
import { useTranslation } from "react-i18next";
import { Linking, NativeModules, SectionList } from "react-native";
import { List, Portal, useTheme } from "react-native-paper";

import { LucideIcon } from "../components";
import { APP_VERSION } from "../const";
import {
  ClearHistoryModal,
  CustomCountryModal,
  DataExportModal,
  DataImportModal,
  DebugModal,
  LicensesModal,
  StatsModal,
} from "../modals";
import { useStore } from "../store";
import { styles } from "../styles";

function SettingsListItemIcon({ name, color }) {
  return name && <LucideIcon color={color} name={name} style={{ marginStart: 16 }} />;
}

export function SettingsScene() {
  const theme = useTheme();
  const { t } = useTranslation();

  const [clearHistoryModal, setClearHistoryModal] = useState(false);
  const [customCountryModal, setCustomCountryModal] = useState(false);
  const [dataExportModal, setDataExportModal] = useState(false);
  const [dataImportModal, setDataImportModal] = useState(false);
  const [debugModal, setDebugModal] = useState(false);
  const [licensesModal, setLicensesModal] = useState(false);
  const [statsModal, setStatsModal] = useState(false);

  const customCountry = useStore((state) => state.customCountry);
  const stats = useStore((state) => state.stats);

  const menu = [
    {
      title: t("settings_section_main"),
      data: [
        {
          icon: "Headphones",
          title: t("settings_audio_effects"),
          onPress: () => NativeModules.AudioEffects.openAudioEffects(),
        },
        {
          icon: "BadgePlus",
          title: t("settings_send_usage_stats"),
          description: stats
            ? t("settings_send_usage_stats_enabled")
            : t("settings_send_usage_stats_disabled"),
          onPress: () => setStatsModal(true),
        },
      ],
    },
    {
      title: t("settings_section_data_management"),
      data: [
        {
          icon: "Trash2",
          title: t("settings_clear_history"),
          onPress: () => setClearHistoryModal(true),
        },
        {
          icon: "Download",
          title: t("settings_data_export"),
          onPress: () => setDataExportModal(true),
        },
        {
          icon: "Upload",
          title: t("settings_data_import"),
          onPress: () => setDataImportModal(true),
        },
      ],
    },
    {
      title: t("settings_section_advanced"),
      data: [
        {
          icon: "Globe",
          title: t("settings_custom_country"),
          description: customCountry || t("settings_custom_country_auto"),
          onPress: () => setCustomCountryModal(true),
        },
        {
          icon: "BatteryLow",
          title: t("settings_prevent_deep_sleep"),
          description: t("settings_prevent_deep_sleep_description"),
          onPress: () => NativeModules.BatteryOptimization.openBatteryOptimizationSettings(),
        },
      ],
    },
    {
      title: t("settings_section_about"),
      data: [
        {
          icon: "SquareArrowOutUpRight",
          title: "Demodulate",
          description: `v${APP_VERSION}`,
          onPress: () => Linking.openURL("https://mateusznowak.dev/demodulate"),
        },
        {
          icon: "Shapes",
          title: t("settings_credits_oss"),
          onPress: () => setLicensesModal(true),
        },
        {
          onPress: () => setDebugModal(true),
        },
      ],
    },
  ];

  return (
    <>
      <SectionList
        renderItem={({ item, key }) => (
          <List.Item
            data={item}
            key={key}
            left={() => <SettingsListItemIcon color={theme.colors.primary} name={item.icon} />}
            {...item}
          />
        )}
        renderSectionHeader={({ section }) => (
          <List.Subheader style={{ color: theme.colors.primary }}>{section.title}</List.Subheader>
        )}
        sections={menu}
        style={{ marginBottom: styles.scene.marginBottom }}
      />
      <Portal>
        {clearHistoryModal && <ClearHistoryModal hide={() => setClearHistoryModal(false)} />}
        {customCountryModal && <CustomCountryModal hide={() => setCustomCountryModal(false)} />}
        {dataExportModal && <DataExportModal hide={() => setDataExportModal(false)} />}
        {dataImportModal && <DataImportModal hide={() => setDataImportModal(false)} />}
        {debugModal && <DebugModal hide={() => setDebugModal(false)} />}
        {licensesModal && <LicensesModal hide={() => setLicensesModal(false)} />}
        {statsModal && <StatsModal hide={() => setStatsModal(false)} />}
      </Portal>
    </>
  );
}
