import AsyncStorage from "@react-native-async-storage/async-storage";
import { createDocument, openDocument } from "react-native-scoped-storage";

import { MAX_ENTRIES } from "../const";
import { getLogger } from "../logging";
import { useStore } from "../store";
import { getServerList, getStationsByUUIDs, sendPlaybackStats } from "./RadioBrowser";

const KNOWN_KEYS = [
  "bookmarks",
  "bookmarksNextUpdate",
  "customCountry",
  "history",
  "historyNextUpdate",
  "player",
  "serverList",
  "serversNextUpdate",
  "stats",
];

const log = getLogger("LocalStorage");

export function cleanData(data) {
  return {
    id: data.id,
    name: data.name,
    url: data.url,
    artwork: data.artwork,
    codec: data.codec,
    valid: data.valid,
    timestamp: data.timestamp,
  };
}

function getFutureDate(hours, minutes, seconds) {
  const d = new Date();
  d.setHours(d.getHours() + hours);
  d.setMinutes(d.getMinutes() + minutes);
  d.setSeconds(d.getSeconds() + seconds);

  return d;
}

async function _loadStore() {
  return await AsyncStorage.multiGet(KNOWN_KEYS).then((data) =>
    Object.fromEntries(data.filter(([, v]) => v).map(([k, v]) => [k, JSON.parse(v)])),
  );
}

async function _saveStore(data) {
  return await AsyncStorage.multiSet(
    Object.entries(data)
      .filter(([k, v]) => KNOWN_KEYS.includes(k) && v !== undefined)
      .map(([k, v]) => [k, JSON.stringify(v)]),
  );
}

export async function loadStore() {
  if (useStore.getState().ready) {
    return;
  }

  log.i("load store");
  await _loadStore()
    .then((data) => {
      useStore.setState(data);
    })
    .catch((err) => {
      log.e(`load store fail msg=${err}`);
    })
    .finally(() => {
      useStore.setState({ ready: true });
    });
}

export async function updatePlayer(data) {
  let player = useStore.getState().player;
  const oldId = player.id;
  const newId = data?.id;

  player = { ...player, ...(data || {}) };

  useStore.setState({ player });

  // Update stored data only if station ID has changed. Do not store temporary metadata.
  if (newId && oldId !== newId) {
    log.i("save player data");
    await _saveStore({ player: cleanData(player) }).catch((err) => {
      log.e(`save player data fail msg=${err}`);
    });
  }

  return player;
}

export async function addBookmark(data) {
  let bookmarks = useStore.getState().bookmarks;

  data = { ...cleanData(data), timestamp: +new Date(), valid: true };

  const index = bookmarks.map((b) => b.id).indexOf(data.id);
  if (index >= 0) {
    return;
  }

  bookmarks = [data, ...bookmarks.slice(0, MAX_ENTRIES - 1)];

  useStore.setState({ bookmarks });

  log.i("save bookmark data");
  await _saveStore({ bookmarks }).catch((err) => {
    log.e(`save bookmark data fail msg=${err}`);
  });
}

export async function removeBookmark(data) {
  let bookmarks = useStore.getState().bookmarks;

  const index = bookmarks.map((b) => b.id).indexOf(data.id);
  if (index < 0) {
    return;
  }

  bookmarks = bookmarks.toSpliced(index, 1);

  useStore.setState({ bookmarks });

  log.i("save bookmark data");
  await _saveStore({ bookmarks }).catch((err) => {
    log.e(`save bookmark data fail msg=${err}`);
  });
}

export async function addHistory(data) {
  let history = useStore.getState().history;

  data = { ...cleanData(data), timestamp: +new Date(), valid: true };

  const index = history.map((h) => h.id).indexOf(data.id);
  if (index >= 0) {
    history = [data, ...history.toSpliced(index, 1).slice(0, MAX_ENTRIES)];
  } else {
    history = [data, ...history.slice(0, MAX_ENTRIES - 1)];
  }

  useStore.setState({ history });

  log.i("save history data");
  await _saveStore({ history }).catch((err) => {
    log.e(`save history data fail msg=${err}`);
  });

  await sendPlaybackStats(data.id);
}

export async function clearHistory() {
  useStore.setState({ history: [] });

  log.i("save history data");
  await _saveStore({ history: [] }).catch((err) => {
    log.e(`save history data fail msg=${err}`);
  });
}

export async function refreshServerList() {
  const serversNextUpdate = useStore.getState().serversNextUpdate;
  if (serversNextUpdate > +new Date()) {
    log.i("server list up to date");
    return;
  }

  const serverList = await getServerList().catch(() => []);
  if (serverList.length === 0) {
    log.i("server list empty, retrying later");
    return;
  }
  await setServerList(serverList);
}

export async function refreshBookmarks() {
  const bookmarksNextUpdate = useStore.getState().bookmarksNextUpdate;
  if (bookmarksNextUpdate > +new Date()) {
    log.i("bookmarks meta up to date");
    return;
  }

  let bookmarks = useStore.getState().bookmarks;
  const newBookmarks = await getStationsByUUIDs(bookmarks.map((b) => b.id)).catch(() => []);
  const newBookmarksMap = new Map(newBookmarks.map((s) => [s.id, s]));

  // Data may be stale, reload and process immediately. Keep original add timestamps.
  bookmarks = useStore.getState().bookmarks.map((b) => {
    const b2 = newBookmarksMap.get(b.id);
    if (!b2) {
      return b;
    }

    return { ...b2, timestamp: b.timestamp };
  });

  await setBookmarks(bookmarks);
}

export async function refreshHistory() {
  const historyNextUpdate = useStore.getState().historyNextUpdate;
  if (historyNextUpdate > +new Date()) {
    log.i("history meta up to date");
    return;
  }

  let history = useStore.getState().history;
  const newHistory = await getStationsByUUIDs(history.map((h) => h.id)).catch(() => []);
  const newHistoryMap = new Map(newHistory.map((s) => [s.id, s]));

  // Data may be stale, reload and process immediately. Keep original add timestamps.
  history = useStore.getState().history.map((h) => {
    const h2 = newHistoryMap.get(h.id);
    if (!h2) {
      return h;
    }

    return { ...h2, timestamp: h.timestamp };
  });

  await setHistory(history);
}

async function setServerList(serverList) {
  // Refresh no more often than every 1 hour.
  const data = { serverList, serversNextUpdate: +getFutureDate(1, 0, 0) };

  useStore.setState(data);

  log.i(`save server list data=${serverList}`);
  await _saveStore(data).catch((err) => {
    log.e(`save server list fail msg=${err}`);
  });
}

export async function setBookmarks(bookmarks) {
  // Refresh no more often than every 2 hours.
  const data = { bookmarks, bookmarksNextUpdate: +getFutureDate(2, 0, 0) };

  useStore.setState(data);

  log.i("save bookmarks meta");
  await _saveStore(data).catch((err) => {
    log.e(`save bookmarks meta fail msg=${err}`);
  });
}

export async function setHistory(history) {
  // Refresh no more often than every 2 hours.
  const data = { history, historyNextUpdate: +getFutureDate(2, 0, 0) };

  useStore.setState(data);

  log.i("save history meta");
  await _saveStore(data).catch((err) => {
    log.e(`save history meta fail msg=${err}`);
  });
}

export async function setStats(stats) {
  useStore.setState({ stats });

  log.i("save stats preference");
  await _saveStore({ stats }).catch((err) => {
    log.e(`save stats preference fail msg=${err}`);
  });
}

export async function setCustomCountry(customCountry) {
  customCountry = customCountry.normalize().toLowerCase().trim().substring(0, 2);

  useStore.setState({ customCountry });

  log.i("save country preference");
  await _saveStore({ customCountry }).catch((err) => {
    log.e(`save country preference fail msg=${err}`);
  });
}

export async function importFromScopedStorage() {
  return await openDocument(true)
    .then(async (result) => {
      if (result?.name) {
        const data = JSON.parse(result.data);
        const bookmarks = data.bookmarks.map(cleanData);
        const history = data.history.map(cleanData);

        await setBookmarks(bookmarks);
        await setHistory(history);

        return { status: "success", name: result.name };
      } else {
        return { status: "canceled" };
      }
    })
    .catch(() => ({ status: "failure" }));
}

export async function exportToScopedStorage() {
  const data = JSON.stringify({
    bookmarks: useStore.getState().bookmarks,
    history: useStore.getState().history,
  });

  return await createDocument("demodulate-data.json", "application/json", data)
    .then((result) => {
      if (result?.name) {
        return { status: "success", name: result.name };
      } else {
        return { status: "canceled" };
      }
    })
    .catch(() => ({ status: "failure" }));
}
