import { useEffect, useState } from "react";
import { StatusBar, View, useColorScheme } from "react-native";
import { BottomNavigation, PaperProvider, Portal, useTheme } from "react-native-paper";
import { useSafeAreaInsets } from "react-native-safe-area-context";

import { loadStore, refreshBookmarks, refreshHistory, refreshServerList } from "./api/LocalStorage";
import { LucideIcon, Player } from "./components";
import { StatsModal } from "./modals";
import { sceneMap, sceneRoutes } from "./scenes";
import { useStore } from "./store";
import { styles } from "./styles";

const renderScene = BottomNavigation.SceneMap(sceneMap);

function getContentColor(colorScheme) {
  return colorScheme === "dark" ? "light-content" : "dark-content";
}

function AppInner() {
  const colorScheme = useColorScheme();
  const safeAreaInsets = useSafeAreaInsets();
  const theme = useTheme();

  const [currentTab, setCurrentTab] = useState(1); // bookmarks tab
  const [statsModal, setStatsModal] = useState(false);

  const ready = useStore((state) => state.ready);
  const stats = useStore((state) => state.stats);

  useEffect(() => {
    loadStore();
  }, []);

  useEffect(() => {
    if (ready) {
      refreshServerList();

      if (stats !== true && stats !== false) {
        // probably unset
        setStatsModal(true);
      }
      if (stats) {
        refreshBookmarks();
        refreshHistory();
      }
    }
  }, [ready]);

  if (!ready) {
    return <></>;
  }

  return (
    <View style={{ backgroundColor: theme.colors.background, flexGrow: 1 }}>
      <StatusBar
        backgroundColor={theme.colors.background}
        barStyle={getContentColor(colorScheme)}
      />
      <BottomNavigation
        barStyle={{ height: styles.navbar.height + safeAreaInsets.bottom }}
        keyboardHidesNavigationBar={false}
        navigationState={{ index: currentTab, routes: sceneRoutes }}
        onIndexChange={(id) => setCurrentTab(id)}
        renderScene={renderScene}
        sceneAnimationEnabled
        sceneAnimationType="shifting"
        style={{ paddingTop: safeAreaInsets.top }}
      />
      <Player />
      {statsModal && (
        <Portal>
          <StatsModal hide={() => setStatsModal(false)} />
        </Portal>
      )}
    </View>
  );
}

export function App() {
  return (
    <PaperProvider settings={{ icon: (props) => <LucideIcon {...props} /> }}>
      <AppInner />
    </PaperProvider>
  );
}
