"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.categorizeLicense = categorizeLicense;
exports.analyzeLicenses = analyzeLicenses;
const licenses_1 = require("../constants/licenses");
const LicenseCategory_1 = require("./LicenseCategory");
const descriptions_1 = require("./descriptions");
/**
 * Categorizes a license based on its copyleft characteristics.
 * @param licenseType the license type
 * @returns the license category
 */
function categorizeLicense(licenseType) {
    if (!licenseType || licenseType === 'unknown') {
        return LicenseCategory_1.LicenseCategory.UNKNOWN;
    }
    // check for strong copyleft licenses
    if (licenses_1.STRONG_COPYLEFT_LICENSES_LOWERCASE.has(licenseType.toLowerCase())) {
        return LicenseCategory_1.LicenseCategory.STRONG_COPYLEFT;
    }
    // check for weak copyleft licenses
    if (licenses_1.WEAK_COPYLEFT_LICENSES_LOWERCASE.has(licenseType.toLowerCase())) {
        return LicenseCategory_1.LicenseCategory.WEAK_COPYLEFT;
    }
    // everything else is considered permissive
    return LicenseCategory_1.LicenseCategory.PERMISSIVE;
}
/**
 * Analyzes license data and returns comprehensive statistics.
 * @param report the licenses report data
 * @returns the license analysis result
 */
function analyzeLicenses(report) {
    const byCategory = {
        [LicenseCategory_1.LicenseCategory.STRONG_COPYLEFT]: 0,
        [LicenseCategory_1.LicenseCategory.WEAK_COPYLEFT]: 0,
        [LicenseCategory_1.LicenseCategory.PERMISSIVE]: 0,
        [LicenseCategory_1.LicenseCategory.UNKNOWN]: 0,
    };
    const byLicense = {};
    const categorizedLicenses = {};
    Object.entries(report).forEach(([packageKey, license]) => {
        const licenseType = license.type;
        const category = categorizeLicense(licenseType);
        // stats by category
        byCategory[category]++;
        // stats by specific license
        {
            const key = licenseType ?? 'unknown';
            byLicense[key] = (byLicense[key] ?? 0) + 1;
        }
        // memoization for lookup
        categorizedLicenses[packageKey] = category;
    });
    const total = Object.keys(report).length;
    const { categoriesPresence, description } = (0, descriptions_1.getGraphStateInfo)(byCategory);
    return {
        total,
        byCategory,
        byLicense,
        description,
        categoriesPresence,
        categorizedLicenses,
    };
}
