package dev.mateusznowak.demodulate

import com.facebook.react.ReactPackage
import com.facebook.react.bridge.Arguments
import com.facebook.react.bridge.NativeModule
import com.facebook.react.bridge.Promise
import com.facebook.react.bridge.ReactApplicationContext
import com.facebook.react.bridge.ReactContextBaseJavaModule
import com.facebook.react.bridge.ReactMethod
import com.facebook.react.uimanager.ViewManager
import okhttp3.OkHttpClient
import okhttp3.Request
import java.io.BufferedInputStream
import java.nio.charset.Charset
import java.util.concurrent.TimeUnit

typealias Result = Pair<String, String>

class MediaTypeDetectorModule(reactContext: ReactApplicationContext) :
    ReactContextBaseJavaModule(reactContext) {

    override fun getName(): String {
        return "MediaTypeDetector"
    }

    @ReactMethod
    fun detect(url: String, userAgent: String, timeout: Int, promise: Promise) {
        Thread {
            try {
                val (url, type) = detectInner(url, userAgent, timeout)
                promise.resolve(Arguments.createMap().apply {
                    putString("url", url)
                    putString("type", type)
                })
            } catch (e: Exception) {
                promise.reject(e)
            }
        }.start()
    }

    private fun detectInner(url: String, userAgent: String, timeout: Int): Result {
        val client = OkHttpClient.Builder().callTimeout(timeout.toLong(), TimeUnit.MILLISECONDS)
            .followRedirects(true).followSslRedirects(true).build()

        val request = Request.Builder().url(url).header("User-Agent", userAgent).get().build()

        client.newCall(request).execute().use { response ->
            val buffer = ByteArray(2048)

            val input = BufferedInputStream(response.body?.byteStream())
            val read = input.read(buffer)
            if (read <= 0) {
                return Result(url, "default")
            }

            val text = buffer.copyOf(read).toString(Charset.forName("UTF-8")).lowercase()

            val detectors: List<() -> Result?> = listOf(
                { tryDetectPls(url, text) },
                { tryDetectM3u(url, text) },
                { tryDetectDash(url, text) },
                { tryDetectSs(url, text) })

            for (detector in detectors) {
                val result = detector()
                if (result != null) {
                    return result
                }
            }

            return Result(url, "default")
        }
    }

    private fun tryDetectPls(url: String, data: String): Result? {
        if ("file" !in data) {
            return null
        }

        val foundUrl = data.lineSequence().map { it.trim() }
            .mapNotNull { Regex("""^file\d+=(.+)$""").find(it)?.groupValues?.get(1) }.toList()
            .takeIf { it.isNotEmpty() }?.firstOrNull()

        return if (foundUrl != null) Result(foundUrl, "default") else null
    }

    private fun tryDetectM3u(url: String, data: String): Result? {
        if ("#extm3u" !in data) {
            return null
        }

        // Negative or 0-length segments may indicate this is not HLS but a regular playlist
        val foundUrl = data.lineSequence().map { it.trim() }.zipWithNext()
            .firstOrNull { (line, _) -> line.startsWith("#extinf:-1") || line.startsWith("#extinf:0") }?.second

        return if (foundUrl != null) Result(foundUrl, "default") else Result(url, "hls")
    }

    private fun tryDetectDash(url: String, data: String): Result? {
        return if ("<mpd" in data) Result(url, "dash") else null
    }

    private fun tryDetectSs(url: String, data: String): Result? {
        return if ("<smooth" in data) Result(url, "smoothstreaming") else null
    }
}

class MediaTypeDetectorPackage : ReactPackage {
    override fun createNativeModules(reactContext: ReactApplicationContext): List<NativeModule> {
        return listOf(MediaTypeDetectorModule(reactContext))
    }

    override fun createViewManagers(reactContext: ReactApplicationContext): List<ViewManager<*, *>> {
        return emptyList()
    }
}
