import { useEffect, useRef, useState } from "react";
import { useTranslation } from "react-i18next";
import { AppState, View } from "react-native";
import { Searchbar, Text } from "react-native-paper";
import { useDebounce } from "use-debounce";

import { getStationsByName, getStationsLocal } from "../api/RadioBrowser";
import { getCountryCode } from "../api/common";
import { StationList } from "../components";
import { useStore } from "../store";
import { styles } from "../styles";

export function SearchScene() {
  const searchbarRef = useRef(null);
  const { t } = useTranslation();

  const [query, setQuery] = useState("");
  const [queryDebounced] = useDebounce(query, 500);

  const [data, setData] = useState([]);
  const [error, setError] = useState(null);
  const [loading, setLoading] = useState(false);

  const server = useStore((state) => state.serverCurrent);

  function search() {
    setLoading(true);

    getCountryCode()
      .then((code) => {
        const changeServer = !!error;
        if (queryDebounced) {
          return getStationsByName(queryDebounced, changeServer);
        } else {
          return getStationsLocal(code, changeServer);
        }
      })
      .then((data) => {
        setData(data);
        setError(null);
      })
      .catch((err) => {
        setData([]);
        setError(err);
      })
      .finally(() => {
        setLoading(false);
      });
  }

  useEffect(() => {
    // WORKAROUND: Scenes are lazy loaded and never unmounted, so the search bar may get stuck in
    // focus mode and then trigger the onscreen keyboard when app is restored.
    AppState.addEventListener("change", (next) => {
      if (next === "background") {
        searchbarRef.current?.blur();
      }
    });
  }, []);

  useEffect(search, [queryDebounced]);

  return (
    <View style={{ flexGrow: 1, flexShrink: 1, marginTop: 8 }}>
      <Searchbar
        clearIcon="X"
        icon="Search"
        inputStyle={{ marginHorizontal: 12 }}
        onChangeText={setQuery}
        placeholder={t("search")}
        ref={searchbarRef}
        style={{ marginHorizontal: 16 }}
        value={query}
      />
      <View style={{ marginHorizontal: 16 }}>
        <Text numberOfLines={1} style={{ opacity: 0.5, textAlign: "center" }}>
          {server}
        </Text>
      </View>
      <StationList
        data={data}
        emptyIcon="FileQuestionMark"
        emptyText={t("search_empty")}
        error={error}
        loading={loading}
        retry={search}
        role="search"
        style={{ flexGrow: 1, marginBottom: styles.scene.marginBottom }}
      />
    </View>
  );
}
