import { useState } from "react";
import { Image } from "react-native";
import { IconButton, List, Portal, useTheme } from "react-native-paper";

import i18next from "../i18n";
import { RemoveBookmarkModal } from "../modals";
import { playTrack } from "../services/PlaybackService";

function getDescription(data, isSearch, isHistory, isPlaying) {
  const t = i18next.t;
  const chunks = [];

  if (isPlaying) {
    chunks.push(t("station_playing"));
  }

  if (isSearch) {
    if (data.hls) {
      chunks.push(t("station_hls"));
    } else if (data.codec) {
      chunks.push(data.codec);
    }
  } else if (isHistory) {
    chunks.push(new Date(data.timestamp).toLocaleDateString());
  }

  if (!data.valid) {
    chunks.push(t("station_broken"));
  }

  return chunks.join(" - ");
}

function StationListItemButton({ data, role }) {
  const theme = useTheme();

  const [removeData, setRemoveData] = useState(null);

  if (role !== "bookmarks") {
    return;
  }

  return (
    <>
      <IconButton
        icon="StarOff"
        iconColor={theme.colors.primary}
        onPress={() => setRemoveData(data)}
        style={{ height: 48, marginRight: -16, marginVertical: 0, padding: 0, width: 48 }}
      />
      {removeData && (
        <Portal>
          <RemoveBookmarkModal data={removeData} hide={() => setRemoveData(null)} />
        </Portal>
      )}
    </>
  );
}

export function StationListItem({ data, player, role }) {
  const theme = useTheme();

  const isSelected = data.id === player.id;
  const isPlaying = isSelected && player.playing;

  const isSearch = role === "search";
  const isHistory = role === "history";

  return (
    <List.Item
      description={getDescription(data, isSearch, isHistory, isPlaying)}
      descriptionNumberOfLines={1}
      left={(props) => <Image {...props} height={48} source={{ uri: data.artwork }} width={48} />}
      onPress={() => playTrack(data)}
      right={() => <StationListItemButton data={data} role={role} />}
      title={data.name}
      titleStyle={isSelected ? { color: theme.colors.primary } : {}}
    />
  );
}
