import axios from "axios";
import TrackPlayer, {
  AppKilledPlaybackBehavior,
  Capability,
  Event,
  State,
} from "react-native-track-player";

import { addHistory, updatePlayer } from "../api/LocalStorage";
import notificationIcon from "../assets/notification.png";
import { NET_TIMEOUT, STREAM_USER_AGENT } from "../const";
import { getLogger } from "../logging";
import { useStore } from "../store";

export const Status = State;

const client = axios.create({
  headers: { "User-Agent": STREAM_USER_AGENT },
  timeout: NET_TIMEOUT,
});

const log = getLogger("PlaybackService");

async function processUrl(url) {
  let isPlaylist = false;

  try {
    const response = await client.head(url);

    // Handle redirects
    url = response.request.responseURL || url;

    // Check whether server has responded with a playlist
    const ctype = response.headers.get("Content-Type");
    if (/x-scpls/.test(ctype)) {
      isPlaylist = true;
    }
  } catch {}

  // Check whether URL looks like it's pointing to a playlist
  isPlaylist = isPlaylist || url.toLowerCase().endsWith(".pls");

  if (!isPlaylist) {
    return url;
  }

  // If it looks like a playlist, fetch its data and get random URL
  try {
    const response = await axios.get(url);

    const urls = response.data
      .split(/\r?\n/)
      .map((line) => line.match(/^File\d+=(.+)$/i)?.[1])
      .filter((u) => u);

    url = urls[+new Date() % urls.length] || url;
  } catch {}

  return url;
}

export async function setupPlayer() {
  const caps = [
    Capability.Play,
    Capability.Pause,
    Capability.SkipToPrevious,
    Capability.SkipToNext,
  ];

  if (!(await TrackPlayer.isServiceRunning())) {
    log.i("initialize TrackPlayer");
    await TrackPlayer.setupPlayer();
  }

  await TrackPlayer.updateOptions({
    android: {
      appKilledPlaybackBehavior: AppKilledPlaybackBehavior.PausePlayback,
      alwaysPauseOnInterruption: true,
    },
    capabilities: caps,
    compactCapabilities: caps,
    icon: notificationIcon, // works only on release builds, see RNTP docs
  });
}

export async function playTrack(data) {
  data = await updatePlayer(data);
  if (!data.url) {
    return false;
  }

  await pauseTrack();

  log.i("add track");

  const track = {
    // WORKAROUND: Some stream URLs contain playlist data that need to be parsed. Neither ExoPlayer
    // nor RNTP can't do that automatically.
    url: await processUrl(data.url),
    title: data.name,
    artwork: data.artwork,
    type: data.hls ? "hls" : "default",
    // WORKAROUND: Set common user agent to make troublesome streams work.
    userAgent: STREAM_USER_AGENT,
  };

  // TrackPlayer.play() alone will play stale content after pausing.
  // TrackPlayer.reset() crashed when called from the notification area's player.
  // Temporarily removing current track is the only thing that seems to work.
  // Removing track as late as possible, to avoid the notification from disappearing.
  if ((await TrackPlayer.getActiveTrackIndex()) >= 0) {
    log.i("remove track");
    // WORKAROUND: handle fake tracks, see comments below
    await TrackPlayer.remove([2, 1, 0]);
  }

  // WORKAROUND: On older Android versions, "prev" and "next" buttons will be shown only if previous
  // and next tracks are in a queue. There must be two fake objects that will never get used.
  await TrackPlayer.add([track, track, track]);
  await TrackPlayer.play();
  await TrackPlayer.move(2, 0);

  await addHistory(data);
  return true;
}

export async function pauseTrack() {
  if (await TrackPlayer.getActiveTrack()) {
    log.i("pause track");
    await TrackPlayer.pause();
  }

  return true;
}

export async function prevTrack() {
  const player = useStore.getState().player;
  const bookmarks = useStore.getState().bookmarks;

  if (bookmarks.length === 0) {
    return false;
  }

  let index = bookmarks.map((b) => b.id).indexOf(player.id);
  if (index >= 0) {
    index -= 1;
    if (index < 0) {
      index = bookmarks.length - 1;
    }
  } else {
    index = 0;
  }

  await playTrack(bookmarks[index]);
  return true;
}

export async function nextTrack() {
  const player = useStore.getState().player;
  const bookmarks = useStore.getState().bookmarks;

  if (bookmarks.length === 0) {
    return false;
  }

  let index = bookmarks.map((b) => b.id).indexOf(player.id);
  if (index >= 0) {
    index += 1;
    if (index >= bookmarks.length) {
      index = 0;
    }
  } else {
    index = 0;
  }

  await playTrack(bookmarks[index]);
  return true;
}

async function updateNotificationMetadata() {
  const current = useStore.getState().player;

  // WORKAROUND: ignore fake tracks, see comments above
  await TrackPlayer.updateMetadataForTrack(1, {
    title: current.title || current.name,
    artist: current.artist,
    album: current.name,
    artwork: current.artwork,
  });
}

export async function PlaybackService() {
  TrackPlayer.addEventListener(Event.RemotePlay, async () => {
    log.i("event RemotePlay");
    await playTrack();
  });

  TrackPlayer.addEventListener(Event.RemotePause, async () => {
    log.i("event RemotePause");
    await pauseTrack();
  });

  TrackPlayer.addEventListener(Event.RemotePrevious, async () => {
    log.i("event RemotePrevious");
    await prevTrack();
  });

  TrackPlayer.addEventListener(Event.RemoteNext, async () => {
    log.i("event RemoteNext");
    await nextTrack();
  });

  TrackPlayer.addEventListener(Event.RemoteDuck, async (e) => {
    log.i(`event RemoteDuck permanent=${e.permanent}`);

    // Stop the playback and remove player from the notification area.
    if (e.permanent) {
      await TrackPlayer.reset();
    }
  });

  TrackPlayer.addEventListener(Event.MetadataCommonReceived, async (e) => {
    log.i(`event MetadataCommonReceived`);

    const raw = e.metadata?.title;
    const hadTitle = useStore.getState().player.title;

    let title, artist;
    if (raw) {
      [artist, title] = raw?.split(/\s-\s(.*)/s, 2);
      artist = artist?.trim();
      title = title?.trim();
    }

    const hasTitle = title && artist;
    if (hasTitle) {
      await updatePlayer({ title, artist });
    } else {
      await updatePlayer({ title: null, artist: null });
    }

    // I have no idea why displaying artwork is so buggy, but this is the only way to make it
    // consistent. It should keep the artwork displayed if the station does not send useful metadata
    // ever, and will also clear only the title/artist fields during ad breaks.
    if (hasTitle || hadTitle) {
      await updateNotificationMetadata();
    }
  });

  TrackPlayer.addEventListener(Event.PlaybackState, async (e) => {
    log.i(
      `event PlaybackState state=${useStore.getState().player.status}->${e.state} ${e.error?.code || ""}`,
    );

    await updatePlayer({
      status: e.state,
      playing: e.state === Status.Playing,
      error: e.error?.code,
    });
  });
}
