"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.legacyDefaultScanPackageOptionsFactory = void 0;
exports.getPackageJsonPath = getPackageJsonPath;
exports.resolvePackageJsonFromEntry = resolvePackageJsonFromEntry;
exports.findPackageRoot = findPackageRoot;
exports.normalizePackageName = normalizePackageName;
exports.prepareAboutLibrariesLicenseField = prepareAboutLibrariesLicenseField;
exports.parseAuthorField = parseAuthorField;
exports.parseLicenseField = parseLicenseField;
exports.parseRepositoryFieldToUrl = parseRepositoryFieldToUrl;
const fs_1 = __importDefault(require("fs"));
const path_1 = __importDefault(require("path"));
const repositoryUtils_1 = require("../../utils/repositoryUtils");
const miscUtils_1 = require("./miscUtils");
function getPackageJsonPath(dependency, root) {
    const rootsToSearch = [
        ...(root ? [root] : []), // provided for purpose of nested node_modules resolution inside a package dir, in a subdirectory inside the root node_modules dir
        process.cwd(), // fallback - root node_modules directory home
    ]; // in order of priority (left-to-right)
    try {
        return require.resolve(`${dependency}/package.json`, { paths: rootsToSearch });
    }
    catch (error) {
        for (const root of rootsToSearch) {
            const pkgJsonInNodeModules = path_1.default.join(root, 'node_modules', dependency, 'package.json');
            if (fs_1.default.existsSync(pkgJsonInNodeModules)) {
                return pkgJsonInNodeModules;
            }
        }
        return resolvePackageJsonFromEntry(dependency); // final fallback
    }
}
function resolvePackageJsonFromEntry(dependency) {
    try {
        const entryPath = require.resolve(dependency);
        const packageDir = findPackageRoot(entryPath);
        if (!packageDir)
            return null;
        const packageJsonPath = path_1.default.join(packageDir, 'package.json');
        return fs_1.default.existsSync(packageJsonPath) ? packageJsonPath : null;
    }
    catch {
        return null;
    }
}
function findPackageRoot(entryPath) {
    let currentDir = path_1.default.dirname(entryPath);
    while (currentDir !== path_1.default.dirname(currentDir)) {
        if (fs_1.default.existsSync(path_1.default.join(currentDir, 'package.json')))
            return currentDir;
        currentDir = path_1.default.dirname(currentDir);
    }
}
function normalizePackageName(packageName) {
    return packageName.replace('/', '_');
}
function prepareAboutLibrariesLicenseField(license) {
    if (!license.type) {
        return '';
    }
    return `${license.type}_${(0, miscUtils_1.sha512)(license.content ?? license.type)}`;
}
function parseAuthorField(json) {
    if (typeof json.author === 'object' && typeof json.author.name === 'string') {
        return json.author.name;
    }
    if (typeof json.author === 'string') {
        return json.author;
    }
}
function parseLicenseField(json) {
    if (typeof json.license === 'object' && typeof json.license.type === 'string') {
        return json.license.type;
    }
    if (typeof json.license === 'string') {
        return json.license;
    }
}
function parseRepositoryFieldToUrl(json) {
    if (typeof json.repository === 'object' && typeof json.repository.url === 'string') {
        return (0, repositoryUtils_1.normalizeRepositoryUrl)(json.repository.url);
    }
    if (typeof json.repository === 'string') {
        return (0, repositoryUtils_1.normalizeRepositoryUrl)(json.repository);
    }
}
/**
 * Default value consistent with legacy behaviour assumptions for the scan package options factory
 * used so as not to introduce breaking API changes
 */
const legacyDefaultScanPackageOptionsFactory = () => ({
    includeTransitiveDependencies: true,
    includeDevDependencies: false,
    includeOptionalDependencies: true,
});
exports.legacyDefaultScanPackageOptionsFactory = legacyDefaultScanPackageOptionsFactory;
