import { Logger } from '../logger'
import { addMissingSpaces, isNumber, isValidJSValue, pipe, roundToPrecision, smartSplit } from './common'

const parseStringValue = (value: string) => {
    if (isValidJSValue(value)) {
        return value
    }

    if (value.startsWith('function')) {
        return value
    }

    const tokens = smartSplit(addMissingSpaces(value))
    const parsedTokens = tokens.map(token => {
        // String literals
        if (token.startsWith('"')) {
            return token.replace(/"/g, '')
        }

        // Plain words
        if (/^[a-z]+$/i.test(token.replace(/,/g, ''))) {
            return token
        }

        // Numbers
        if (isNumber(token)) {
            return token
        }

        // Expressions that need to be wrapped with ${}
        const endsWithComma = token.endsWith(',')

        return [
            '${',
            endsWithComma ? token.slice(0, -1) : token,
            '}',
            endsWithComma ? ',' : '',
        ].join('')
    })

    return [
        '`',
        parsedTokens.join(' '),
        '`',
    ].join('')
}

export const serialize = (value: any): string => {
    const typeOfValue = typeof value

    switch (typeOfValue) {
        case 'string':
            return parseStringValue(value)
        case 'symbol':
            return String(value)
        case 'number':
        case 'bigint':
            return String(roundToPrecision(value, 2))
        case 'boolean':
            return value.toString()
        case 'object':
            if (value === null) {
                return 'null'
            }

            if (Array.isArray(value)) {
                return [
                    '[',
                    value.map(serialize).join(', '),
                    ']',
                ].join('')
            }

            return [
                '{',
                Object.entries(value).map(([key, val]) => {
                    const serializedKey = isNumber(key) ? key : `"${key}"`

                    return `${serializedKey}: ${serialize(val)}`
                }).join(', '),
                '}',
            ].join('')
        case 'undefined':
            return 'undefined'
        case 'function':
            return ''
        default:
            typeOfValue satisfies never
            return ''
    }
}

export const serializeJSObject = (obj: Record<string, any>, serializer: (key: string, value: string) => string) => {
    const serializedObject = pipe(obj)(
        Object.entries,
        entries => entries.map(([key, value]) => serializer(key, serialize(value))),
        entries => entries.join(','),
        result => {
            if (result === '') {
                return ''
            }

            return `${result},`
        },
    )

    try {
        // eslint-disable-next-line @typescript-eslint/no-implied-eval, no-new-func
        new Function(`function validateJS() { const obj = ({ ${serializedObject} }) }`)
    } catch {
        Logger.error('Failed to create virtual js')
        return ''
    }

    return serializedObject
}
