"use strict";

import React, { memo } from 'react';
import { useSharedProps } from "./context/SharedPropsProvider.js";
import { Text } from '@native-html/transient-render-engine';
import useAssembledCommonProps from "./hooks/useAssembledCommonProps.js";
import { useTNodeChildrenRenderer } from "./context/TChildrenRendererContext.js";
import renderTextualContent from "./renderTextualContent.js";
import { useRendererRegistry } from "./context/RenderRegistryProvider.js";
import renderBlockContent from "./renderBlockContent.js";
import renderEmptyContent from "./renderEmptyContent.js";
import domNodeToHTMLString from "./helpers/domNodeToHTMLString.js";
const TDefaultBlockRenderer = renderBlockContent.bind(null);
TDefaultBlockRenderer.displayName = 'TDefaultBlockRenderer';
const TDefaultPhrasingRenderer = renderTextualContent.bind(null);
TDefaultPhrasingRenderer.displayName = 'TDefaultPhrasingRenderer';
const TDefaultTextRenderer = renderTextualContent.bind(null);
TDefaultTextRenderer.displayName = 'TDefaultTextRenderer';
function isGhostTNode(tnode) {
  return tnode.type === 'text' && (tnode.data === '' || tnode.data === ' ') || tnode.type === 'empty';
}

/**
 * A component to render any {@link TNode}.
 */
const TNodeRenderer = /*#__PURE__*/memo(function MemoizedTNodeRenderer(props) {
  const {
    tnode
  } = props;
  const sharedProps = useSharedProps();
  const renderRegistry = useRendererRegistry();
  const TNodeChildrenRenderer = useTNodeChildrenRenderer();
  const tnodeProps = {
    propsFromParent: {
      collapsedMarginTop: null
    },
    ...props,
    TNodeChildrenRenderer,
    sharedProps
  };
  const renderer = tnode.type === 'block' || tnode.type === 'document' ? TDefaultBlockRenderer : tnode.type === 'text' ? TDefaultTextRenderer : tnode.type === 'phrasing' ? TDefaultPhrasingRenderer : renderEmptyContent;
  const {
    assembledProps,
    Renderer
  } = useAssembledCommonProps(tnodeProps, renderer);
  switch (tnode.type) {
    case 'empty':
      if (tnode.isUnregistered) {
        const text = domNodeToHTMLString(tnode.domNode);
        return renderTextualContent({
          ...assembledProps,
          type: 'text',
          tnode: {
            ...tnode,
            type: 'text',
            data: text,
            textNode: new Text(text),
            getReactNativeProps: () => ({})
          }
        });
      } else {
        return renderEmptyContent(assembledProps);
      }
    case 'text':
      const InternalTextRenderer = renderRegistry.getInternalTextRenderer(props.tnode.tagName);
      if (InternalTextRenderer) {
        return /*#__PURE__*/React.createElement(InternalTextRenderer, tnodeProps);
      }
      // If ghost line prevention is enabled and the text data is empty, render
      // nothing to avoid React Native painting a 20px height line.
      // See also https://git.io/JErwX
      if (tnodeProps.tnode.data === '' && tnodeProps.tnode.tagName == null && tnodeProps.sharedProps.enableExperimentalGhostLinesPrevention) {
        return null;
      }
      break;
    case 'phrasing':
      // When a TPhrasing node is anonymous and has only one child, its
      // rendering amounts to rendering its only child.
      if (tnodeProps.sharedProps.bypassAnonymousTPhrasingNodes && tnodeProps.tnode.tagName == null && tnodeProps.tnode.children.length <= 1) {
        return /*#__PURE__*/React.createElement(TNodeChildrenRenderer, {
          tnode: props.tnode
        });
      }
      // If ghost line prevention is enabled and the tnode is an anonymous empty
      // phrasing node, render nothing to avoid React Native painting a 20px
      // height line. See also https://git.io/JErwX
      if (tnodeProps.sharedProps.enableExperimentalGhostLinesPrevention && tnodeProps.tnode.tagName == null && tnodeProps.tnode.children.every(isGhostTNode)) {
        return null;
      }
      break;
  }
  if (Renderer !== null) {
    return /*#__PURE__*/React.createElement(Renderer, assembledProps);
  }
  if (tnode.type === 'block' || tnode.type === 'document') {
    return renderBlockContent(assembledProps);
  }
  return renderTextualContent(assembledProps);
});
export { TDefaultBlockRenderer, TDefaultPhrasingRenderer, TDefaultTextRenderer };
export default TNodeRenderer;
//# sourceMappingURL=TNodeRenderer.js.map