"use strict";

import equals from 'ramda/src/equals';
import React, { memo, useMemo } from 'react';
import { Dimensions } from 'react-native';
import ttreeEventsContext from "./context/ttreeEventsContext.js";
import isUriSource from "./helpers/isUriSource.js";
import SourceLoaderUri from "./SourceLoaderUri.js";
import SourceLoaderInline from "./SourceLoaderInline.js";
import SourceLoaderDom from "./SourceLoaderDom.js";
import debugMessage from "./debugMessages.js";
import contentWidthContext from "./context/contentWidthContext.js";
import isDomSource from "./helpers/isDomSource.js";
import useProfiler from "./hooks/useProfiler.js";
import { jsx as _jsx } from "react/jsx-runtime";
function isEmptySource(source) {
  return !source || typeof source.uri !== 'string' && typeof source.html !== 'string' && !source.dom;
}
function RawSourceLoader({
  source,
  ...props
}) {
  if (isEmptySource(source)) {
    /* istanbul ignore next */
    if (typeof __DEV__ === 'boolean' && __DEV__) {
      console.warn(debugMessage.noSource);
    }
    return null;
  }
  if (isUriSource(source)) {
    return /*#__PURE__*/React.createElement(SourceLoaderUri, {
      source,
      ...props
    });
  }
  if (isDomSource(source)) {
    return /*#__PURE__*/React.createElement(SourceLoaderDom, {
      source,
      ...props
    });
  }
  return /*#__PURE__*/React.createElement(SourceLoaderInline, {
    source,
    ...props
  });
}

// check if for each key of two objects, the values are equal
function shallowEqual(prop1, prop2) {
  if (!equals(Object.keys(prop1), Object.keys(prop2))) {
    return false;
  }
  for (const key in prop1) {
    if (prop1[key] !== prop2[key]) {
      return false;
    }
  }
  return true;
}

/**
 * A React component to render HTML snippets.
 *
 * @remarks This component is useful when you have to load dozens of HTML
 * snippets with the same config. Performance is expected to improve in such
 * scenarios.
 *
 * @warning This component requires to have {@link TRenderEngineProvider}
 * and {@link RenderHTMLConfigProvider} as parents.
 */
const RenderHTMLSource = /*#__PURE__*/memo(function RenderHtmlSource({
  onDocumentMetadataLoaded,
  onTTreeChange,
  contentWidth,
  ...props
}) {
  const profile = useProfiler({
    prop: 'onDocumentMetadataLoaded or onTTreeChange'
  });
  const ttreeEvents = useMemo(() => {
    typeof __DEV__ === 'boolean' && __DEV__ && profile();
    return {
      onDocumentMetadataLoaded,
      onTTreeChange
    };
  }, [onDocumentMetadataLoaded, onTTreeChange, profile]);
  if (typeof __DEV__ === 'boolean' && __DEV__) {
    if (!(typeof contentWidth === 'number')) {
      console.warn(debugMessage.contentWidth);
    }
  }
  return /*#__PURE__*/_jsx(ttreeEventsContext.Provider, {
    value: ttreeEvents,
    children: /*#__PURE__*/_jsx(contentWidthContext.Provider, {
      value: contentWidth || Dimensions.get('window').width,
      children: /*#__PURE__*/React.createElement(RawSourceLoader, props)
    })
  });
}, ({
  source: prevSource,
  ...prev
}, {
  source: currSource,
  ...curr
}) => {
  return shallowEqual(prevSource, currSource) && shallowEqual(prev, curr);
});
export default RenderHTMLSource;
//# sourceMappingURL=RenderHTMLSource.js.map