#!/usr/bin/env node
"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.downloadProfile = downloadProfile;
var _cliTools = require("@react-native-community/cli-tools");
var _fs = _interopRequireDefault(require("fs"));
var _path = _interopRequireDefault(require("path"));
var _os = _interopRequireDefault(require("os"));
var _child_process = require("child_process");
var _hermesProfileTransformer = _interopRequireDefault(require("@margelo/hermes-profile-transformer"));
var _getMetroBundleOptions = require("./getMetroBundleOptions");
var _sourcemapUtils = require("./sourcemapUtils");
var _getConfig = _interopRequireDefault(require("./getConfig"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
// Most of the file is just a copy of https://github.com/react-native-community/cli/blob/main/packages/cli-hermes/src/profileHermes/downloadProfile.ts

/**
 * Get the last modified hermes profile
 * @param packageNameWithSuffix
 */
function getLatestFile(packageNameWithSuffix) {
  try {
    const file = (0, _child_process.execSync)(`adb shell run-as ${packageNameWithSuffix} ls cache/ -tp | grep -v /$ | grep -E '.cpuprofile' | head -1
        `);
    return file.toString().trim();
  } catch (e) {
    throw e;
  }
}
function execSyncWithLog(command) {
  _cliTools.logger.debug(`${command}`);
  return (0, _child_process.execSync)(command);
}

/**
 * A wrapper that converts an object to JSON with 4 spaces for indentation.
 *
 * @param obj Any object that can be represented as JSON
 * @returns A JSON string
 */
function jsonStringify(obj) {
  return JSON.stringify(obj, undefined, 4);
}
function getLatestFileFromDownloads() {
  try {
    const file = (0, _child_process.execSync)(`adb shell ls "/sdcard/Download" -tp | grep -v /$ | grep -E '.cpuprofile' | head -1`);
    return file.toString().trim();
  } catch (e) {
    throw e;
  }
}
function maybeAddLineAndColumn(path) {
  const profile = _fs.default.readFileSync(path, 'utf8');
  const profileJson = JSON.parse(profile);
  const stackFrames = profileJson.stackFrames;
  if (stackFrames !== undefined) {
    for (const key of Object.keys(stackFrames)) {
      const stackFrame = stackFrames[key];
      if (stackFrame.funcVirtAddr && stackFrame.offset) {
        stackFrame.line = `${1}`;
        stackFrame.column = `${parseInt(stackFrame.funcVirtAddr) + parseInt(stackFrame.offset) + 1}`;
        delete stackFrame.funcVirtAddr;
        delete stackFrame.offset;
      }
    }
    _fs.default.writeFileSync(path, JSON.stringify(profileJson));
  }
}

/**
 * Pull and convert a Hermes tracing profile to Chrome tracing profile
 * @param ctx
 * @param dstPath
 * @param fileName
 * @param sourceMapPath
 * @param raw
 * @param generateSourceMap
 * @param appId
 * @param appIdSuffix
 */
async function downloadProfile(local, fromDownload, dstPath, filename, sourcemapPath, raw, shouldGenerateSourcemap, port = '8081', appId, appIdSuffix) {
  let ctx = await (0, _getConfig.default)();
  try {
    const androidProject = ctx === null || ctx === void 0 ? void 0 : ctx.project.android;
    const packageNameWithSuffix = [appId || (androidProject === null || androidProject === void 0 ? void 0 : androidProject.packageName), appIdSuffix].filter(Boolean).join('.');
    if (!packageNameWithSuffix && !local) {
      throw new Error("Failed to retrieve the package name from the project's Android manifest file. Please provide the package name with the --appId flag.");
    }

    // If file name is not specified, pull the latest file from device
    let file = filename || (fromDownload ? getLatestFileFromDownloads() : getLatestFile(packageNameWithSuffix));
    if (!file) {
      throw new _cliTools.CLIError('There is no file in the cache/ directory. Did you record a profile from the developer menu?');
    }
    _cliTools.logger.info(`File to be pulled: ${file}`);

    // If destination path is not specified, pull to the current directory
    dstPath = dstPath || (ctx === null || ctx === void 0 ? void 0 : ctx.root) || '.';
    _cliTools.logger.debug('Internal commands run to pull the file:');

    // If --raw, pull the hermes profile to dstPath
    if (raw) {
      if (fromDownload) {
        execSyncWithLog(`adb shell cat /sdcard/Download/${file} > ${dstPath}/${file}`);
      } else {
        execSyncWithLog(`adb shell run-as ${packageNameWithSuffix} cat cache/${file} > ${dstPath}/${file}`);
      }
      maybeAddLineAndColumn(`${dstPath}/${file}`);
      _cliTools.logger.success(`Successfully pulled the file to ${dstPath}/${file}`);
    }

    // Else: transform the profile to Chrome format and pull it to dstPath
    else {
      const osTmpDir = _os.default.tmpdir();
      const tempFilePath = _path.default.join(osTmpDir, file);
      if (local) {
        _fs.default.copyFileSync(local, tempFilePath);
      } else if (fromDownload) {
        execSyncWithLog(`adb shell cat /sdcard/Download/${file} > ${tempFilePath}`);
      } else {
        execSyncWithLog(`adb shell run-as ${packageNameWithSuffix} cat cache/${file} > ${tempFilePath}`);
      }
      maybeAddLineAndColumn(tempFilePath);
      const bundleOptions = (0, _getMetroBundleOptions.getMetroBundleOptions)(tempFilePath, 'localhost');

      // If path to source map is not given
      if (!sourcemapPath) {
        // Get or generate the source map
        if (shouldGenerateSourcemap) {
          sourcemapPath = await (0, _sourcemapUtils.generateSourcemap)(port, bundleOptions);
        } else {
          sourcemapPath = await (0, _sourcemapUtils.findSourcemap)(ctx, port, bundleOptions);
        }

        // Run without source map
        if (!sourcemapPath) {
          _cliTools.logger.warn('Cannot find source maps, running the transformer without it');
          _cliTools.logger.info('Instructions on how to get source maps: set `bundleInDebug: true` in your app/build.gradle file, inside the `project.ext.react` map.');
        }
      }

      // Run transformer tool to convert from Hermes to Chrome format
      const events = await (0, _hermesProfileTransformer.default)(tempFilePath, sourcemapPath, 'index.bundle');
      const transformedFilePath = `${dstPath}/${_path.default.basename(file, '.cpuprofile')}-converted.json`;

      // Convert to JSON in chunks because JSON.stringify() will fail for large
      // arrays with the error "RangeError: Invalid string length"
      const out = events.map(jsonStringify).join(',');
      _fs.default.writeFileSync(transformedFilePath, '[' + out + ']', 'utf-8');
      _cliTools.logger.success(`Successfully converted to Chrome tracing format and pulled the file to ${transformedFilePath}`);
    }
  } catch (e) {
    throw e;
  }
}
function getFilename(path) {
  if (path == null) {
    return null;
  }
  const nodes = path.split('/');
  const res = nodes[nodes.length - 1];
  return res;
}
const {
  program
} = require('commander');
program.option('--filename <string>').option('--sourcemap-path <string>').option('--generate-sourcemap').option('--port <number>').option('--appId <string>').option('--appIdSuffix <string>').option('--fromDownload').option('--raw').option('--local <string>');
program.parse();
const options = program.opts();
const dstPath = '.';
downloadProfile(options.local, options.fromDownload, dstPath, options.filename || getFilename(options.local), options.sourcemapPath, options.raw, options.generateSourcemap, options.port, options.appId, options.appIdSuffix);
//# sourceMappingURL=cli.js.map