"use strict";

import { useMemo } from 'react';
import getDimensionsWithAspectRatio from "./getDimensionsWithAspectRatio.js";
import { StyleSheet } from 'react-native';
function normalizeSize(dimension, options) {
  const containerDimension = options.containerDimension || null;
  const enablePercentWidth = options.enablePercentWidth || false;
  if (dimension === null || dimension === undefined || Number.isNaN(dimension)) {
    return null;
  }
  if (typeof dimension === 'number') {
    return dimension;
  }
  if (dimension.search('%') !== -1 && enablePercentWidth && typeof containerDimension === 'number') {
    return parseFloat(dimension) * containerDimension / 100;
  } else if (dimension.trim().match(/^[\d.]+$/)) {
    return parseFloat(dimension);
  }
  return null;
}

/**
 * Extract specified dimensions from props.
 */
function deriveSpecifiedDimensionsFromProps({
  width,
  height,
  contentWidth,
  flatStyle,
  enableExperimentalPercentWidth: enablePercentWidth
}) {
  const normalizeOptionsWidth = {
    enablePercentWidth,
    containerDimension: contentWidth
  };
  const normalizeOptionsHeight = {
    enablePercentWidth: false
  };
  const widthProp = normalizeSize(width, normalizeOptionsWidth);
  const heightProp = normalizeSize(height, normalizeOptionsHeight);
  const styleWidth = normalizeSize(flatStyle.width, normalizeOptionsWidth);
  const styleHeight = normalizeSize(flatStyle.height, normalizeOptionsHeight);
  return getDimensionsWithAspectRatio(styleWidth ?? widthProp, styleHeight ?? heightProp, flatStyle.aspectRatio);
}
export default function useImageSpecifiedDimensions(props) {
  const {
    contentWidth,
    enableExperimentalPercentWidth = false,
    style = {},
    width,
    height
  } = props;
  const flatStyle = useMemo(() => StyleSheet.flatten(style) || {}, [style]);
  const specifiedDimensions = useMemo(() => deriveSpecifiedDimensionsFromProps({
    contentWidth,
    enableExperimentalPercentWidth,
    width,
    height,
    flatStyle
  }), [contentWidth, enableExperimentalPercentWidth, flatStyle, height, width]);
  return {
    flatStyle,
    specifiedDimensions
  };
}
//# sourceMappingURL=useImageSpecifiedDimensions.js.map