import { requireNativeModule } from "expo";
import { PermissionsAndroid } from "react-native";
/**
 * Subscribe to the messages sent from the distributor background service.
 * This callback will be called whenever the distributor sends a message to the device.
 * It will not run if the app is terminated but notifications will still be delivered even if this callback is not registered.
 * It will also receive error messages created when displaying notifications on the background service.
 *
 * @param fn The function that will receive the distributor messages.
 *
 * This function will always receive an object with the properties `action` and `data`:
 * - `data`: The data sent from the distributor. Varies depending on the `action` property.
 * - `action`: The action that was performed by the distributor. Can be one of the following:
 * - - `"message"`: A push notification was received. Data for this type is {@link MessagePayload}.
 *        The text in `data.message` will contain the JSON-encoded payload of the push notification if `data.decrypted` is `true`.
 *        Otherwise, it will contain the raw encrypted payload.
 * - - `"registered"`: The device has been registered with a distributor. Data for this type is {@link RegisteredPayload}. This is the data that is needed for the backend to send a push notification to a specific device.
 * - - `"registrationFailed"`: The device failed to register with a distributor. Data for this type is {@link RegistrationFailedPayload}.
 * - - `"unregistered"`: The device has been unregistered from a distributor. Data for this type is {@link UnregisteredPayload}.
 * - - `"error"`: An error occurred while receiving a message from a distributor. Data for this type is {@link ErrorPayload}.
 */
export function subscribeDistributorMessages(fn) {
    const listener = module.addListener("message", fn);
    return () => listener.remove();
}
/**
 * Request the necessary permissions for the distributor to send messages to the device.
 * This method will request the `POST_NOTIFICATIONS` android permission.
 * If this module is running inside an emulator, this function will always return `"denied"`, as notifications are not supported on emulators.
 *
 * @returns A promise that resolves to 'granted' if the permission was granted, 'denied' if the user denied the permission, or 'never_ask_again' if the user chose to never ask again.
 */
export function requestPermissions() {
    // @ts-ignore: Private field access is intentional
    if (module.__isEmulator()) {
        return Promise.resolve("denied");
    }
    return PermissionsAndroid.request(PermissionsAndroid.PERMISSIONS.POST_NOTIFICATIONS);
}
/**
 * Check if the `POST_NOTIFICATIONS` android permission has been granted.
 * If this module is running inside an emulator, this function will always return `false`, as notifications are not supported on emulators
 *
 * @returns A promise that resolves to `true` if the permission has been granted, `false` otherwise.
 */
export function checkPermissions() {
    // @ts-ignore: Private field access is intentional
    if (module.__isEmulator()) {
        return Promise.resolve(false);
    }
    return PermissionsAndroid.check(PermissionsAndroid.PERMISSIONS.POST_NOTIFICATIONS);
}
/**
 * Show a local notification. This is a helper function that will show a notification using the native notification API.
 * It can be used to test the different types of notifications that can be received from the distributor.
 * The same payload that this function receives is the one that the backend will have to send to the distributor.
 *
 * @param notification The notification object with all its options.
 */
export async function showLocalNotification(notification) {
    try {
        const json = JSON.stringify(notification);
        // @ts-ignore: Private field access is intentional
        await module.__showLocalNotification(json);
    }
    catch (error) {
        console.error(error);
    }
}
// This call loads the native module object from the JSI.
const module = requireNativeModule("ExpoUnifiedPush");
export default module;
//# sourceMappingURL=ExpoUnifiedPushModule.js.map