import { EventObject, EventPayloadMap, ExtractEvents, StoreConfig, StoreContext, StoreLogic, StoreSnapshot } from "./types.js";
interface UndoRedoEventOptions<TContext extends StoreContext, TEvent extends EventObject> {
    /** A function that returns the transaction ID of an event. */
    getTransactionId?: (event: TEvent, snapshot: StoreSnapshot<TContext>) => string | null | undefined;
    /**
     * A function that returns whether an event should be skipped during
     * undo/redo. Skipped events are not stored in history and are not replayed
     * during undo/redo.
     */
    skipEvent?: (event: TEvent, snapshot: StoreSnapshot<TContext>) => boolean;
}
interface UndoRedoSnapshotOptions<TContext extends StoreContext, TEvent extends EventObject> {
    /** A function that returns the transaction ID of an event. */
    getTransactionId?: (event: TEvent, snapshot: StoreSnapshot<TContext>) => string | null | undefined;
    /**
     * A function that returns whether a snapshot should be skipped during
     * undo/redo. Skipped events don't save snapshots to history.
     */
    skipEvent?: (event: TEvent, snapshot: StoreSnapshot<TContext>) => boolean;
    /** Maximum number of snapshots to keep in history. Defaults to Infinity. */
    historyLimit?: number;
    /**
     * A function to compare snapshots for equality. When true, the new snapshot
     * will not be added to history. Useful for avoiding duplicate snapshots.
     */
    compare?: (pastSnapshot: StoreSnapshot<TContext>, currentSnapshot: StoreSnapshot<TContext>) => boolean;
}
type UndoRedoStrategyOptions<TContext extends StoreContext, TEvent extends EventObject> = ({
    strategy?: 'event';
} & UndoRedoEventOptions<TContext, TEvent>) | ({
    strategy: 'snapshot';
} & UndoRedoSnapshotOptions<TContext, TEvent>);
/**
 * Creates store logic with undo/redo functionality.
 *
 * Supports two strategies:
 *
 * - 'event' (default): Maintains event history and replays events
 * - 'snapshot': Maintains snapshot history for faster undo/redo
 *
 * @example
 *
 * ```ts
 * // Event strategy (default)
 * const store = createStore(
 *   undoRedo({
 *     context: { count: 0 },
 *     on: {
 *       inc: (ctx) => ({ count: ctx.count + 1 })
 *     }
 *   })
 * );
 * ```
 *
 * @example
 *
 * ```ts
 * // Snapshot strategy
 * const store = createStore(
 *   undoRedo(
 *     {
 *       context: { count: 0 },
 *       on: {
 *         inc: (ctx) => ({ count: ctx.count + 1 })
 *       }
 *     },
 *     {
 *       strategy: 'snapshot',
 *       historyLimit: 10
 *     }
 *   )
 * );
 * ```
 *
 * @returns Store logic with additional `undo` and `redo` event handlers
 */
export declare function undoRedo<TContext extends StoreContext, TEventPayloadMap extends EventPayloadMap, TEmittedPayloadMap extends EventPayloadMap>(storeConfig: StoreConfig<TContext, TEventPayloadMap, TEmittedPayloadMap>, options?: UndoRedoStrategyOptions<TContext, ExtractEvents<TEventPayloadMap>>): StoreLogic<StoreSnapshot<TContext>, ExtractEvents<TEventPayloadMap> | {
    type: 'undo';
} | {
    type: 'redo';
}, ExtractEvents<TEmittedPayloadMap>>;
export {};
