import {
  useLoginMutation,
  useLoginMfaMutation,
  DEFAULT_INSTANCE,
  useEnvironment,
} from '@/lib/api/auth'
import { Link } from 'expo-router'
import { useState } from 'react'
import { TextInput, View, Text } from 'react-native'
import { useThemeColor } from '@/hooks/useThemeColor'
import { Colors } from '@/constants/Colors'
import InstanceProvider from '@/components/InstanceProvider'
import { useToasts } from '@/lib/toasts'
import Button from './Button'
import { KeyboardAwareScrollView } from 'react-native-keyboard-controller'
import { useAuth } from '@/lib/contexts/AuthContext'

export default function ModalSignIn({
  onLoginComplete,
}: {
  onLoginComplete: (token: string, instance: string) => void
}) {
  const [email, setEmail] = useState('')
  const [password, setPassword] = useState('')
  const [mfaToken, setMfaToken] = useState('')
  const [firstPassToken, setFirstPassToken] = useState('')

  const { instance: savedInstance } = useAuth()
  const [_instance, setInstance] = useState<string | null>(null)
  const instance = _instance ?? savedInstance ?? DEFAULT_INSTANCE

  const { data: env, status, isLoading } = useEnvironment(instance)
  const envStatus = isLoading ? 'pending' : status

  const { showToastError } = useToasts()

  const color = useThemeColor({}, 'text')
  const loginMfaMutation = useLoginMfaMutation(env)
  const loginMutation = useLoginMutation(env)

  async function completeLogin(token: string) {
    onLoginComplete(token, instance)
  }

  function login() {
    if (env) {
      loginMutation.mutate(
        { email, password },
        {
          onSuccess: async (firstPassResponse) => {
            if (firstPassResponse.mfaRequired) {
              setFirstPassToken(firstPassResponse.token)
            } else {
              await completeLogin(firstPassResponse.token)
            }
          },
          onError: (error) => {
            console.error(error)
            showToastError('Invalid credentials')
          },
        },
      )
    }
  }

  function loginMfa() {
    if (env) {
      loginMfaMutation.mutate(
        { firstPassToken, mfaToken },
        {
          onSuccess: (token) => completeLogin(token),
          onError: (error) => {
            console.error(error)
            showToastError('Invalid credentials')
            setMfaToken('')
            setFirstPassToken('')
          },
        },
      )
    }
  }

  return (
    <View
      className="flex-1 px-4 pt-5"
      style={{ backgroundColor: Colors.dark.background }}
    >
      <KeyboardAwareScrollView>
        <InstanceProvider
          savedInstance={instance}
          setSavedInstance={setInstance}
          envStatus={envStatus}
        >
          {!firstPassToken && (
            <>
              <View className="mt-3">
                <TextInput
                  inputMode="email"
                  autoCapitalize="none"
                  autoCorrect={false}
                  placeholder="Email"
                  style={{ color }}
                  className="p-3 my-3 border border-gray-500 rounded placeholder:text-gray-400"
                  value={email}
                  onChangeText={setEmail}
                />
                <TextInput
                  secureTextEntry
                  autoCapitalize="none"
                  placeholder="Password"
                  style={{ color }}
                  className="p-3 my-3 border border-gray-500 rounded placeholder:text-gray-400"
                  value={password}
                  onChangeText={setPassword}
                />
                <Link href="/password-reset" className="text-blue-500 pb-3">
                  Forgot your password?
                </Link>
              </View>
              <View className="py-3">
                <Button
                  disabled={
                    loginMutation.isPending || !env || !email || !password
                  }
                  text={loginMutation.isPending ? 'Loading...' : 'Sign in'}
                  onPress={login}
                />
              </View>
              <Text className="py-3 text-white">
                {"Don't"} have an account?{' '}
                <Link href="/register" className="text-blue-500">
                  Register here
                </Link>
              </Text>
            </>
          )}
          {firstPassToken && (
            <>
              <Text className="py-3 text-white">
                Please enter your token generated by your Authenticator
                application
              </Text>
              <View className="mt-3">
                <TextInput
                  inputMode="numeric"
                  autoCapitalize="none"
                  placeholder="Token"
                  style={{ color }}
                  className="p-3 my-3 border border-gray-500 rounded placeholder:text-gray-400"
                  value={mfaToken}
                  onChangeText={setMfaToken}
                />
                <Link href="/password-reset" className="text-blue-500 pb-3">
                  Lost your device?
                </Link>
              </View>
              <View className="py-3">
                <Button
                  disabled={loginMfaMutation.isPending || !env || !mfaToken}
                  text={loginMfaMutation.isPending ? 'Loading...' : 'Sign in'}
                  onPress={loginMfa}
                />
              </View>
            </>
          )}
        </InstanceProvider>
      </KeyboardAwareScrollView>
    </View>
  )
}
