import { Dimensions, Platform } from "react-native";
import { Orientation, StyleDependency } from "../../types.js";
import { cloneWithAccessors } from "./native-utils.js";
import { parseBoxShadow, parseFontVariant, parseTransformsMutation, resolveGradient } from "./parsers/index.js";
import { UniwindRuntime } from "./runtime.js";
class UniwindStoreBuilder {
  runtime = UniwindRuntime;
  vars = {};
  stylesheet = {};
  listeners = {
    [StyleDependency.ColorScheme]: /* @__PURE__ */ new Set(),
    [StyleDependency.Theme]: /* @__PURE__ */ new Set(),
    [StyleDependency.Dimensions]: /* @__PURE__ */ new Set(),
    [StyleDependency.Orientation]: /* @__PURE__ */ new Set(),
    [StyleDependency.Insets]: /* @__PURE__ */ new Set(),
    [StyleDependency.FontScale]: /* @__PURE__ */ new Set(),
    [StyleDependency.Rtl]: /* @__PURE__ */ new Set()
  };
  hotReloadListeners = /* @__PURE__ */ new Set();
  cache = /* @__PURE__ */ new Map();
  generateStyleSheetCallbackResult = null;
  subscribe(onStoreChange, dependencies) {
    if (__DEV__) {
      this.hotReloadListeners.add(onStoreChange);
    }
    dependencies.forEach((dep) => {
      this.listeners[dep].add(onStoreChange);
    });
    return () => {
      dependencies.forEach((dep) => {
        this.listeners[dep].delete(onStoreChange);
      });
      if (__DEV__) {
        this.hotReloadListeners.delete(onStoreChange);
      }
    };
  }
  getStyles(className, state) {
    if (className === void 0 || className === "") {
      return {
        styles: {},
        dependencies: []
      };
    }
    const cacheKey = `${className}${state?.isDisabled ?? false}${state?.isFocused ?? false}${state?.isPressed ?? false}`;
    if (this.cache.has(cacheKey)) {
      return this.cache.get(cacheKey);
    }
    const result = this.resolveStyles(className, state);
    this.cache.set(cacheKey, result);
    const cacheReset = () => {
      this.cache.delete(cacheKey);
      result.dependencies.forEach((dep) => this.listeners[dep].delete(cacheReset));
    };
    this.subscribe(cacheReset, result.dependencies);
    return result;
  }
  reinit = (generateStyleSheetCallback) => {
    const config = generateStyleSheetCallback?.(this.runtime) ?? this.generateStyleSheetCallbackResult;
    if (!config) {
      return;
    }
    const { scopedVars, stylesheet, vars } = config;
    this.generateStyleSheetCallbackResult = config;
    this.stylesheet = stylesheet;
    this.vars = vars;
    const themeVars = scopedVars[`__uniwind-theme-${this.runtime.currentThemeName}`];
    const platformVars = scopedVars[`__uniwind-platform-${Platform.OS}`];
    if (themeVars) {
      Object.defineProperties(this.vars, Object.getOwnPropertyDescriptors(themeVars));
    }
    if (platformVars) {
      Object.defineProperties(this.vars, Object.getOwnPropertyDescriptors(platformVars));
    }
    if (__DEV__ && generateStyleSheetCallback) {
      this.hotReloadListeners.forEach((listener) => listener());
    }
  };
  notifyListeners = (dependencies) => {
    dependencies.forEach((dep) => this.listeners[dep].forEach((listener) => listener()));
  };
  resolveStyles(classNames, state) {
    const result = {};
    let vars = this.vars;
    const dependencies = [];
    const bestBreakpoints = /* @__PURE__ */ new Map();
    for (const className of classNames.split(" ")) {
      if (!(className in this.stylesheet)) {
        continue;
      }
      for (const style of this.stylesheet[className]) {
        dependencies.push(...style.dependencies);
        if (style.minWidth > this.runtime.screen.width || style.maxWidth < this.runtime.screen.height || style.theme !== null && this.runtime.currentThemeName !== style.theme || style.orientation !== null && this.runtime.orientation !== style.orientation || style.rtl !== null && this.runtime.rtl !== style.rtl || style.active !== null && state?.isPressed !== style.active || style.focus !== null && state?.isFocused !== style.focus || style.disabled !== null && state?.isDisabled !== style.disabled) {
          continue;
        }
        for (const [property, valueGetter] of style.entries) {
          const previousBest = bestBreakpoints.get(property);
          if (previousBest && (previousBest.minWidth > style.minWidth || previousBest.complexity > style.complexity || previousBest.importantProperties.includes(property))) {
            continue;
          }
          if (property[0] === "-") {
            if (vars === this.vars) {
              vars = cloneWithAccessors(this.vars);
            }
            Object.defineProperty(vars, property, {
              configurable: true,
              enumerable: true,
              get: valueGetter
            });
          } else {
            Object.defineProperty(result, property, {
              configurable: true,
              enumerable: true,
              get: () => valueGetter.call(vars)
            });
          }
          bestBreakpoints.set(property, style);
        }
      }
    }
    if (result.lineHeight !== void 0 && result.lineHeight < 6) {
      Object.defineProperty(result, "lineHeight", {
        value: result.fontSize * result.lineHeight,
        configurable: true,
        enumerable: true
      });
    }
    if (result.boxShadow !== void 0) {
      Object.defineProperty(result, "boxShadow", {
        value: parseBoxShadow(result.boxShadow),
        configurable: true,
        enumerable: true
      });
    }
    if (result.visibility === "hidden") {
      Object.defineProperty(result, "display", {
        value: "none",
        configurable: true,
        enumerable: true
      });
    }
    if (result.borderStyle !== void 0 && result.borderColor === void 0) {
      Object.defineProperty(result, "borderColor", {
        value: "#000000",
        configurable: true,
        enumerable: true
      });
    }
    if (result.outlineStyle !== void 0 && result.outlineColor === void 0) {
      Object.defineProperty(result, "outlineColor", {
        value: "#000000",
        configurable: true,
        enumerable: true
      });
    }
    if (result.fontVariant !== void 0) {
      Object.defineProperty(result, "fontVariant", {
        value: parseFontVariant(result.fontVariant),
        configurable: true,
        enumerable: true
      });
    }
    parseTransformsMutation(result);
    if (result.experimental_backgroundImage !== void 0) {
      Object.defineProperty(result, "experimental_backgroundImage", {
        value: resolveGradient(result.experimental_backgroundImage),
        configurable: true,
        enumerable: true
      });
    }
    return {
      styles: { ...result },
      dependencies: Array.from(new Set(dependencies))
    };
  }
}
export const UniwindStore = new UniwindStoreBuilder();
Dimensions.addEventListener("change", ({ window }) => {
  const newOrientation = window.width > window.height ? Orientation.Landscape : Orientation.Portrait;
  const orientationChanged = UniwindStore.runtime.orientation !== newOrientation;
  UniwindStore.runtime.screen = {
    width: window.width,
    height: window.height
  };
  UniwindStore.runtime.orientation = newOrientation;
  UniwindStore.notifyListeners([
    ...orientationChanged ? [StyleDependency.Orientation] : [],
    StyleDependency.Dimensions
  ]);
});
