"use strict";

import { useEffect, useState } from 'react';
import { Image } from 'react-native';
import defaultImageInitialDimensions from "./defaultInitialImageDimensions.js";
import useIMGNormalizedSource from "./useIMGNormalizedSource.js";
import useImageConcreteDimensions from "./useImageConcreteDimensions.js";
import { getIMGState } from "./getIMGState.js";
import useImageSpecifiedDimensions from "./useImageSpecifiedDimensions.js";
function getImageSizeAsync({
  uri,
  headers
}) {
  return new Promise((onsuccess, onerror) => {
    const onImageDimensionsSuccess = (width, height) => onsuccess({
      width,
      height
    });
    if (headers) {
      Image.getSizeWithHeaders(uri, headers, onImageDimensionsSuccess, onerror);
    } else {
      Image.getSize(uri, onImageDimensionsSuccess, onerror);
    }
  });
}
function useImageNaturalDimensions(props) {
  const {
    source,
    cachedNaturalDimensions
  } = props;
  const [naturalDimensions, setNaturalDimensions] = useState(cachedNaturalDimensions || null);
  const {
    width: cachedNaturalWidth,
    height: cachedNaturalHeight
  } = cachedNaturalDimensions || {};
  const [error, setError] = useState(null);
  useEffect(function resetOnURIChange() {
    setNaturalDimensions(cachedNaturalWidth != null && cachedNaturalHeight != null ? {
      width: cachedNaturalWidth,
      height: cachedNaturalHeight
    } : null);
    setError(null);
  }, [cachedNaturalHeight, cachedNaturalWidth, source.uri]);
  return {
    onNaturalDimensions: setNaturalDimensions,
    onError: setError,
    naturalDimensions,
    error
  };
}
function useFetchedNaturalDimensions(props) {
  const {
    source,
    cachedNaturalDimensions
  } = props;
  const {
    error,
    naturalDimensions,
    onError,
    onNaturalDimensions
  } = useImageNaturalDimensions(props);
  const hasCachedDimensions = !!cachedNaturalDimensions;
  useEffect(function fetchPhysicalDimensions() {
    let cancelled = false;
    if (source.uri && !hasCachedDimensions) {
      getImageSizeAsync({
        uri: source.uri,
        headers: source.headers
      }).then(dimensions => !cancelled && onNaturalDimensions(dimensions)).catch(e => !cancelled && onError(e || {}));
      return () => {
        cancelled = true;
      };
    }
    return () => {};
  }, [source.uri, source.headers, onNaturalDimensions, onError, hasCachedDimensions]);
  return {
    naturalDimensions,
    error,
    onError,
    onNaturalDimensions
  };
}
function identity(arg) {
  return arg;
}

/**
 * This hook will compute concrete dimensions from image natural dimensions and
 * constraints. It will fetch the image and get its dimensions.
 *
 * @remarks If you know the dimensions beforehand, use
 * {@link useIMGElementStateWithCache} instead to save a network request and
 * prevent a layout shift.
 */
export default function useIMGElementState(props) {
  const {
    alt,
    altColor,
    source,
    contentWidth,
    computeMaxWidth = identity,
    objectFit,
    initialDimensions = defaultImageInitialDimensions,
    cachedNaturalDimensions
  } = props;
  const {
    flatStyle,
    specifiedDimensions
  } = useImageSpecifiedDimensions(props);
  const nomalizedSource = useIMGNormalizedSource({
    specifiedDimensions,
    source
  });
  const {
    naturalDimensions,
    onError,
    error
  } = useFetchedNaturalDimensions({
    source: nomalizedSource,
    specifiedDimensions,
    cachedNaturalDimensions
  });
  const concreteDimensions = useImageConcreteDimensions({
    flatStyle,
    naturalDimensions,
    specifiedDimensions,
    computeMaxWidth,
    contentWidth
  });
  return getIMGState({
    error,
    alt,
    altColor,
    concreteDimensions,
    containerStyle: flatStyle,
    initialDimensions,
    objectFit,
    onError: err => onError(new Error(err)),
    source: nomalizedSource
  });
}
//# sourceMappingURL=useIMGElementState.js.map