"use strict";

import { StyleSheet, View } from 'react-native';
import { MarkedListItem, useMarkedList } from '@jsamr/react-native-li';
import { useTChildrenRenderer } from "../context/TChildrenRendererContext.js";
import { DEFAULT_TEXT_COLOR } from "../constants.js";
import pick from 'ramda/src/pick';
import { useListStyleSpecs } from "../context/ListStyleSpecsProvider.js";
import { jsx as _jsx } from "react/jsx-runtime";
function getStartIndex(tnode) {
  const parsedIndex = tnode.attributes.start ? Number(tnode.attributes.start) : Number.NaN;
  return Number.isNaN(parsedIndex) ? 1 : parsedIndex;
}
const pickMarkerTextStyles = pick(['fontStyle', 'fontSize', 'fontWeight', 'fontFamily', 'fontVariant', 'color', 'lineHeight']);
function extractMarkerTextStyle(tnode) {
  return Object.assign({
    lineHeight: 14 * 1.3,
    fontSize: 14,
    color: DEFAULT_TEXT_COLOR
  }, pickMarkerTextStyles(tnode.styles.nativeTextFlow));
}
export function getMarkerBoxStyle(markerWidth, paddingValue) {
  const markerBoxWidth = typeof markerWidth === 'number' ? typeof paddingValue === 'number' ? Math.max(paddingValue, markerWidth) : markerWidth : paddingValue;
  return {
    width: markerBoxWidth
  };
}
const listStyleTypeFallbackRecord = {
  ol: 'decimal',
  ul: 'disc'
};
export default function ListElement({
  tnode,
  TDefaultRenderer,
  listType,
  style,
  getFallbackListStyleTypeFromNestLevel,
  markerBoxStyle,
  markerTextStyle: providedMarkerTextStyle,
  enableExperimentalRtl = false,
  enableRemoveTopMarginIfNested = true,
  enableRemoveBottomMarginIfNested = true,
  enableDynamicMarkerBoxWidth = false,
  ...props
}) {
  const listStyleSpecs = useListStyleSpecs();
  const markers = tnode.markers;
  const nestLevel = listType === 'ol' ? markers.olNestLevel : markers.ulNestLevel;
  const TChildrenRenderer = useTChildrenRenderer();
  const rtl = enableExperimentalRtl && (style.direction === 'rtl' || markers.direction === 'rtl');
  const removeTopMarginStyle = enableRemoveTopMarginIfNested && tnode.parent?.tagName === 'li' && tnode.nodeIndex === 0 ? styles.zeroMarginTop : null;
  const removeBottomMarginStyle = enableRemoveBottomMarginIfNested && tnode.parent?.tagName === 'li' && tnode.nodeIndex === tnode.parent?.children.length - 1 ? styles.zeroMarginBottom : null;
  const ownListType = tnode.styles.webTextFlow.listStyleType;
  const selectedListType = getFallbackListStyleTypeFromNestLevel(nestLevel) || ownListType || listStyleTypeFallbackRecord[listType];
  const listStyleType = ownListType || selectedListType;
  if (typeof __DEV__ === 'boolean' && __DEV__ && !(listStyleType in listStyleSpecs)) {
    if (listStyleType.match(/^("|')/)) {
      console.warn("This library doesn't support strings for list-style-type CSS properties.");
    } else {
      console.warn(`list-style-type "${listStyleType}" is not handled by react-native-html-engine. ` + 'You can easily provide support for this style via "customListStyleSpecs" prop.');
    }
  }
  const spec = listStyleType in listStyleSpecs ? listStyleSpecs[listStyleType] : listStyleSpecs[selectedListType];
  const counterRenderer = spec.counterStyleRenderer;
  const startIndex = getStartIndex(tnode);
  const markerTextStyle = typeof providedMarkerTextStyle === 'object' && providedMarkerTextStyle ? {
    ...extractMarkerTextStyle(tnode),
    ...providedMarkerTextStyle
  } : extractMarkerTextStyle(tnode);
  const itemProps = useMarkedList({
    counterRenderer,
    startIndex,
    markerTextStyle,
    markerBoxStyle,
    rtlLineReversed: rtl,
    rtlMarkerReversed: rtl,
    length: tnode.children.length,
    dynamicMarkerBoxWidth: enableDynamicMarkerBoxWidth,
    renderMarker: spec.renderMarker
  });
  const markerWidth = itemProps.markerTextWidth;
  const fixedPaddingRule = rtl ? 'paddingRight' : 'paddingLeft';
  // Fallback to padding-left value on RTL mode
  const paddingValue = style[fixedPaddingRule] ?? style.paddingLeft ?? undefined;
  const markerBoxWidthStyle = getMarkerBoxStyle(markerWidth, paddingValue);
  const renderChild = ({
    childElement,
    key,
    index
  }) => /*#__PURE__*/_jsx(MarkedListItem, {
    index: index,
    ...itemProps,
    markerBoxStyle: [itemProps.markerBoxStyle, markerBoxWidthStyle],
    markerTextStyle: itemProps.markerTextStyle,
    enableMarkerClipping: true,
    style: itemProps.style,
    children: /*#__PURE__*/_jsx(View, {
      style: styles.shrink,
      children: childElement
    })
  }, key);
  const dynamicPaddingStyle = {
    position: 'relative',
    [fixedPaddingRule]: 0
  };
  return /*#__PURE__*/_jsx(TDefaultRenderer, {
    tnode: tnode,
    style: [style, removeTopMarginStyle, removeBottomMarginStyle, dynamicPaddingStyle],
    ...props,
    children: /*#__PURE__*/_jsx(TChildrenRenderer, {
      tchildren: tnode.children,
      renderChild: renderChild
    })
  });
}
const styles = StyleSheet.create({
  zeroMarginTop: {
    marginTop: 0
  },
  zeroMarginBottom: {
    marginBottom: 0
  },
  shrink: {
    flexShrink: 1
  }
});
//# sourceMappingURL=ListElement.js.map