"use strict";

import React, { useContext, useEffect, useState } from 'react';
import RenderTTree from "./RenderTTree.js";
import sourceLoaderContext from "./context/sourceLoaderContext.js";
const ERROR_STATE = {
  error: true,
  resolvedHTML: null
};
async function loadHTMLResource(uri, {
  body,
  headers,
  method
}) {
  const response = await fetch(uri, {
    body,
    headers,
    method
  });
  if (response.ok) {
    const html = await response.text();
    return {
      resolvedHTML: html,
      error: false
    };
  }
  return ERROR_STATE;
}
function useUriSourceLoader({
  source,
  onHTMLLoaded
}) {
  const [loadState, setState] = useState({
    error: false,
    resolvedHTML: null
  });
  const {
    error
  } = loadState;

  // Effect to reload on uri changes
  useEffect(() => {
    let cancelled = false;
    if (!error) {
      setState({
        error: false,
        resolvedHTML: null
      });
      loadHTMLResource(source.uri, {
        body: source.body,
        headers: source.headers,
        method: source.method
      }).then(state => {
        !cancelled && setState(state);
      }).catch(() => {
        !cancelled && setState(ERROR_STATE);
      });
    }
    return () => {
      cancelled = true;
    };
  }, [error, source.uri, source.body, source.headers, source.method]);
  useEffect(() => {
    loadState.resolvedHTML && onHTMLLoaded?.call(null, loadState.resolvedHTML);
  }, [loadState.resolvedHTML, onHTMLLoaded]);
  return loadState;
}
export default function SourceLoaderUri(props) {
  const {
    remoteErrorView,
    remoteLoadingView
  } = useContext(sourceLoaderContext);
  const {
    resolvedHTML,
    error
  } = useUriSourceLoader(props);
  if (error) {
    return remoteErrorView.call(null, props.source);
  }
  if (resolvedHTML === null) {
    return remoteLoadingView.call(null, props.source);
  }
  return /*#__PURE__*/React.createElement(RenderTTree, {
    document: resolvedHTML,
    baseUrl: props.source.uri
  });
}
//# sourceMappingURL=SourceLoaderUri.js.map