'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var react = require('react');
var store = require('../../dist/store-db80c133.cjs.js');

function defaultCompare(a, b) {
  return a === b;
}
function identity(snapshot) {
  return snapshot;
}
function useSelectorWithCompare(selector, compare) {
  const previous = react.useRef(undefined);
  return snapshot => {
    const next = selector(snapshot);
    return previous.current && compare(previous.current, next) ? previous.current : previous.current = next;
  };
}

/**
 * A React hook that subscribes to the `store` and selects a value from the
 * store's snapshot via a selector function, with an optional compare function.
 *
 * @example
 *
 * ```ts
 * function Component() {
 *   const count = useSelector(store, (s) => s.count);
 *
 *   return <div>{count}</div>;
 * }
 * ```
 *
 * @param store The store, created from `createStore(…)`
 * @param selector A function which takes in the `snapshot` and returns a
 *   selected value
 * @param compare An optional function which compares the selected value to the
 *   previous value
 * @returns The selected value
 */

/**
 * A React hook that subscribes to the `store` and selects a value from the
 * store's snapshot via an optional selector function (identity by default),
 * with an optional compare function.
 *
 * @example
 *
 * ```ts
 * function Component() {
 *   const countSnapshot = useSelector(store);
 *
 *   return <div>{countSnapshot.context.count}</div>;
 * }
 * ```
 *
 * @param store The store, created from `createStore(…)`
 * @param selector An optional function which takes in the `snapshot` and
 *   returns a selected value
 * @param compare An optional function which compares the selected value to the
 *   previous value
 * @returns The selected value
 */

function useSelector(store, selector = identity, compare = defaultCompare) {
  const selectorWithCompare = useSelectorWithCompare(selector, compare);
  return react.useSyncExternalStore(react.useCallback(handleStoreChange => store.subscribe(handleStoreChange).unsubscribe, [store]), () => selectorWithCompare(store.get()), () => selectorWithCompare(store.get()));
}
const useStore = function useStoreImpl(definition) {
  const storeRef = react.useRef(undefined);
  if (!storeRef.current) {
    storeRef.current = store.createStore(definition);
  }
  return storeRef.current;
};

/**
 * A React hook that subscribes to the `atom` and returns the current value of
 * the atom.
 *
 * @example
 *
 * ```ts
 * const atom = createAtom(0);
 *
 * const Component = () => {
 *   const count = useAtom(atom);
 *
 *   return (
 *     <div>
 *       <div>{count}</div>
 *       <button onClick={() => atom.set((c) => c + 1)}>Increment</button>
 *     </div>
 *   );
 * };
 * ```
 *
 * @param atom The atom, created from `createAtom(…)`
 * @param selector An optional function which takes in the `snapshot` and
 *   returns a selected value
 * @param compare An optional function which compares the selected value to the
 *   previous value
 */

function useAtom(atom, selector = identity, compare = defaultCompare) {
  const state = useSelector(atom, selector, compare);
  return state;
}

/**
 * Creates a custom hook that returns the selected value and the store from a
 * store configuration object.
 *
 * @example
 *
 * ```ts
 * const useCountStore = createStoreHook({
 *   context: { count: 0 },
 *   on: {
 *     inc: (context, event: { by: number }) => ({
 *       ...context,
 *       count: context.count + event.by
 *     })
 *   }
 * });
 *
 * function Component() {
 *   const [count, store] = useCountStore(s => s.context.count);
 *
 *   return (
 *     <div>
 *       {count}
 *       <button onClick={() => store.trigger.inc({ by: 1 })}>+</button>
 *     </div>
 *   );
 * }
 * ```
 *
 * @param definition The store configuration object
 * @returns A custom hook that returns [selectedValue, store]
 */
function createStoreHook(definition) {
  const store$1 = store.createStore(definition);
  function useStoreHook(selector, compare = defaultCompare) {
    // If no selector provided, return full snapshot
    if (!selector) {
      const snapshot = useSelector(store$1, identity, defaultCompare);
      return [snapshot, store$1];
    }

    // Use selector with comparison
    const selectedValue = useSelector(store$1, selector ?? identity, compare);
    return [selectedValue, store$1];
  }
  return useStoreHook;
}

exports.createStoreHook = createStoreHook;
exports.useAtom = useAtom;
exports.useSelector = useSelector;
exports.useStore = useStore;
