"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findMetroPath = findMetroPath;
exports.getMetroVersion = getMetroVersion;
exports.requireModuleFromMetro = requireModuleFromMetro;
const package_1 = require("@rnx-kit/tools-node/package");
const cli_ts_1 = require("./cli.js");
const resolve_ts_1 = require("./resolve.js");
const metroPathCache = {};
function findMetroPathInternal(projectRoot) {
    const rnDir = (0, resolve_ts_1.resolveFrom)("react-native", projectRoot);
    if (!rnDir) {
        return undefined;
    }
    // `metro` dependency was moved to `@react-native/community-cli-plugin` in 0.73
    // https://github.com/facebook/react-native/commit/fdcb94ad1310af6613cfb2a2c3f22f200bfa1c86
    const cliPluginDir = (0, cli_ts_1.findCommunityCliPluginPath)(projectRoot, rnDir);
    if (cliPluginDir) {
        return (0, resolve_ts_1.resolveFrom)("metro", cliPluginDir);
    }
    const cliDir = (0, resolve_ts_1.resolveFrom)("@react-native-community/cli", rnDir);
    if (!cliDir) {
        return undefined;
    }
    const cliMetroDir = (0, resolve_ts_1.resolveFrom)("@react-native-community/cli-plugin-metro", cliDir);
    return (0, resolve_ts_1.resolveFrom)("metro", cliMetroDir || cliDir);
}
/**
 * Finds the installation path of Metro.
 * @param projectRoot The root of the project; defaults to the current working directory
 * @returns The path to the Metro installation; `undefined` if Metro could not be found
 */
function findMetroPath(projectRoot = process.cwd()) {
    if (projectRoot in metroPathCache) {
        return metroPathCache[projectRoot];
    }
    const p = findMetroPathInternal(projectRoot);
    metroPathCache[projectRoot] = p;
    return p;
}
/**
 * Returns Metro version number.
 * @param projectRoot The root of the project; defaults to the current working directory
 * @returns Metro version number; `undefined` if Metro could not be found
 */
function getMetroVersion(projectRoot = process.cwd()) {
    const metroPath = findMetroPath(projectRoot);
    if (!metroPath) {
        return undefined;
    }
    const { version } = (0, package_1.readPackage)(metroPath);
    return version;
}
/**
 * Imports specified module starting from the installation directory of the
 * currently used `metro` version.
 */
function requireModuleFromMetro(moduleName, fromDir = process.cwd()) {
    const startDir = findMetroPath(fromDir);
    if (!startDir) {
        throw new Error("Cannot find module 'metro'");
    }
    const metroDir = "metro/";
    const modulePath = moduleName.startsWith(metroDir)
        ? `${startDir}/${moduleName.substring(metroDir.length)}`
        : (0, resolve_ts_1.resolveFrom)(moduleName, startDir);
    if (!modulePath) {
        throw new Error(`Cannot find module '${moduleName}'. This probably means that ` +
            "'@rnx-kit/tools-react-native' is not compatible with the version " +
            "of 'metro' that you are currently using. Please update to the " +
            "latest version and try again. If the issue still persists after the " +
            "update, please file a bug at " +
            "https://github.com/microsoft/rnx-kit/issues.");
    }
    const m = require(modulePath);
    return m.default ?? m;
}
//# sourceMappingURL=metro.js.map