import * as nodefs from "fs";
/**
 * Components of a package reference.
 */
export type PackageRef = {
    scope?: string;
    name: string;
};
/**
 * Module reference with the package name and optional sub-module path included as path
 */
export type DestructuredModuleRef = PackageRef & {
    path?: string;
};
/**
 * Destructure a module reference into its component par
 * @param r module reference to destructure
 * @returns either a destructured module reference or undefined if it is a file reference
 */
export declare function destructureModuleRef(r: string): DestructuredModuleRef;
/**
 * Parse a package reference string. An example reference is the `name`
 * property found in `package.json`.
 *
 * @param r Package reference string
 * @returns Parsed package reference object
 */
export declare function parsePackageRef(r: string): PackageRef;
/**
 * Schema for a reference to a person in `package.json`.
 */
export type PackagePerson = string | {
    name: string;
    email?: string;
    url?: string;
};
/**
 * Schema for the contents of a `package.json` manifest file.
 */
export type PackageManifest = {
    name: string;
    version: string;
    private?: boolean;
    typings?: string;
    types?: string;
    scripts?: Record<string, string>;
    dependencies?: Record<string, string>;
    peerDependencies?: Record<string, string>;
    devDependencies?: Record<string, string>;
    author?: PackagePerson;
    "rnx-kit"?: Record<string, unknown>;
    [key: string]: string | boolean | string[] | Record<string, unknown> | undefined;
};
/**
 * Read a `package.json` manifest from a file.
 *
 * @param pkgPath Either a path directly to the target `package.json` file, or the directory containing it.
 * @returns Package manifest
 */
export declare function readPackage(pkgPath: string, 
/** @internal */ fs?: typeof nodefs): PackageManifest;
/**
 * Write a `package.json` manifest to a file.
 *
 * @param pkgPath Either a path directly to the target `package.json` file, or the directory containing it.
 * @param manifest Package manifest
 * @param space Indentation to apply to the output
 */
export declare function writePackage(pkgPath: string, manifest: PackageManifest, space?: string, 
/** @internal */ fs?: typeof nodefs): void;
/**
 * Find the nearest `package.json` manifest file. Search upward through all
 * parent directories.
 *
 * If a starting directory is given, use it. Otherwise, use the current working
 * directory.
 *
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to `package.json`, or `undefined` if not found.
 */
export declare function findPackage(startDir?: string, 
/** @internal */ fs?: typeof nodefs): string | undefined;
/**
 * Find the parent directory of the nearest `package.json` manifest file. Search
 * upward through all parent directories.
 *
 * If a starting directory is given, use it. Otherwise, use the current working
 * directory.
 *
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to `package.json`, or `undefined` if not found.
 */
export declare function findPackageDir(startDir?: string, 
/** @internal */ fs?: typeof nodefs): string | undefined;
/**
 * Options which control how package dependecies are located.
 */
export type FindPackageDependencyOptions = {
    /**
     * Optional starting directory for the search. Defaults to `process.cwd()`.
     */
    startDir?: string;
    /**
     * Optional flag controlling whether symlinks can be found. Defaults to `true`.
     * When `false`, and the package dependency directory is a symlink, it will not
     * be found.
     */
    allowSymlinks?: boolean;
    /**
     * Optional flag controlling whether to resolve symlinks. Defaults to `false`.
     * Note that this flag has no effect if `allowSymlinks` is `false`.
     */
    resolveSymlinks?: boolean;
};
/**
 * Find the package dependency's directory, starting from the given directory
 * and moving outward, through all parent directories.
 *
 * Package dependencies exist under 'node_modules/[`scope`]/[`name`]'.
 *
 * @param ref Package dependency reference
 * @param options Options which control the search
 * @returns Path to the package dependency's directory, or `undefined` if not found.
 */
export declare function findPackageDependencyDir(ref: string | PackageRef, options?: FindPackageDependencyOptions, 
/** @internal */ fs?: typeof nodefs): string | undefined;
/**
 * Resolve the path to a dependency given a chain of dependencies leading up to
 * it.
 * @param chain Chain of dependencies leading up to the target dependency.
 * @param startDir Optional starting directory for the search. If not given, the current directory is used.
 * @returns Path to the final dependency's directory.
 */
export declare function resolveDependencyChain(chain: string[], startDir?: string): string;
//# sourceMappingURL=package.d.ts.map