import Loading from '@/components/Loading'
import { BOTTOM_BAR_HEIGHT } from '@/lib/styles'
import useAsyncStorage from '@/lib/useLocalStorage'
import { MaterialCommunityIcons } from '@expo/vector-icons'
import { useState } from 'react'
import { Pressable, Text, TouchableOpacity, View } from 'react-native'
import { KeyboardAwareScrollView } from 'react-native-keyboard-controller'
import { useCSSVariable } from 'uniwind'

export default function SearchIndex({
  onSearch,
}: {
  onSearch: (term: string) => void
}) {
  const gray300 = useCSSVariable('--color-gray-300') as string
  const [showTips, setShowTips] = useState(false)
  const {
    value: recent,
    setValue: setRecent,
    loading,
  } = useAsyncStorage<string[]>('searchHistory', [])

  function toggleTips() {
    setShowTips((f) => !f)
  }

  function removeRecent(item: string) {
    setRecent((recent || []).filter((i) => i !== item))
  }

  return (
    <KeyboardAwareScrollView
      keyboardShouldPersistTaps="always"
      contentContainerStyle={{
        paddingBottom: BOTTOM_BAR_HEIGHT,
      }}
    >
      <View id="search-tips" className="mb-4 p-3">
        <View className="flex-row items-center my-1 gap-1">
          <Text className="text-gray-400 font-medium grow">Search tips</Text>
          <Pressable
            className="py-1 px-3 transform active:scale-95 rounded-full active:bg-blue-600/10"
            onPress={toggleTips}
          >
            <Text className="text-indigo-500 text-sm">
              {showTips ? 'Hide' : 'Show'}
            </Text>
          </Pressable>
        </View>
        {showTips ? (
          <View>
            <Text className="text-white text-sm mb-2">
              You can search local users, remote users, and posts containing a
              hashtag, both in the fediverse and in bluesky.
            </Text>
            <Text className="text-white text-sm mb-2">
              To search for a remote user in the fediverse, enter their full
              username starting with an @ and including the server. As example,
              {'"@torvalds@social.kernel.org"'}
            </Text>
            <Text className="text-white text-sm mb-2">
              To search for a local user or a remote user in bluesky (AT
              Protocol), enter their full username starting with an @. As
              example, {'"@gabboman"'} or {'"@dimension20.bsky.social"'}
            </Text>
            <Text className="text-white text-sm mb-2">
              To search posts with a hashtag, enter the hashtag starting with a
              #. As example, {'"#dnd"'}. You can limit the search to a specific
              user by adding their username before the hashtag. As example,
              {'"user:@alexia #WafrnDev"'}
            </Text>
            <Text className="text-white text-sm mb-2">
              You can also use full URLs to fetch a specific post from a
              specific instance or from bsky.app.
            </Text>
          </View>
        ) : null}
      </View>
      <View id="search-history" className="p-3">
        <View className="mb-2 flex-row items-center gap-1">
          <Text className="text-gray-400 font-medium text-sm grow">
            Search history
          </Text>
          {(recent?.length || 0) > 0 && (
            <Pressable
              onPress={() => setRecent([])}
              className="shrink-0 py-1 px-3 transform active:scale-95 rounded-full active:bg-blue-600/10"
            >
              <Text numberOfLines={1} className="text-indigo-500 text-sm">
                Clear all
              </Text>
            </Pressable>
          )}
        </View>
        {loading && <Loading />}
        {recent?.length === 0 ? (
          <Text className="text-white text-sm">No recent searches</Text>
        ) : (
          recent?.map((item, index) => (
            <TouchableOpacity
              key={index}
              className="flex-row items-center gap-2"
              onPress={() => onSearch(item)}
            >
              <Text className="text-white py-2 grow shrink">{item}</Text>
              <Pressable
                onPress={() => removeRecent(item)}
                className="p-2 shrink-0"
              >
                <MaterialCommunityIcons
                  name="close"
                  size={16}
                  color={gray300}
                />
              </Pressable>
            </TouchableOpacity>
          ))
        )}
      </View>
    </KeyboardAwareScrollView>
  )
}
