import { useEventListener } from 'expo'
import { useVideoPlayer, VideoView } from 'expo-video'
import { useMemo, useState, useEffect } from 'react'
import { Pressable, View, Text, TouchableOpacity } from 'react-native'
import { Ionicons } from '@expo/vector-icons'
import { clsx } from 'clsx'

export default function Video({
  title,
  isAudioOnly = false,
  src,
  width,
  height,
  className,
}: {
  isAudioOnly?: boolean
  src: string
  width: number
  height: number
  title?: string
  className?: string
}) {
  const [showControls, setShowControls] = useState(true)
  const [muted, setMuted] = useState(false)
  const [currentTime, setCurrentTime] = useState(0)
  const [duration, setDuration] = useState(0)
  const [isPlaying, setIsPlaying] = useState(false)

  const source = useMemo(
    () => ({
      uri: src,
      metadata: {
        title: title ?? `WAFRN ${isAudioOnly ? 'audio' : 'video'}`,
      },
    }),
    [src, title, isAudioOnly],
  )

  const player = useVideoPlayer(source, (p) => {
    p.staysActiveInBackground = true
    p.addListener('playingChange', (ev) => {
      if (ev.isPlaying) {
        p.showNowPlayingNotification = true
      }
    })
    p.addListener('playToEnd', () => {
      p.showNowPlayingNotification = false
    })
    p.bufferOptions = {
      preferredForwardBufferDuration: 2, // only buffer the first 2 seconds
    }
    p.loop = true
    p.timeUpdateEventInterval = 1
    return p
  })

  useEventListener(player, 'playingChange', (ev) => setIsPlaying(ev.isPlaying))
  useEventListener(player, 'timeUpdate', (ev) => setCurrentTime(ev.currentTime))
  useEventListener(player, 'sourceLoad', (ev) => setDuration(ev.duration))
  useEventListener(player, 'mutedChange', (ev) => setMuted(ev.muted))

  function togglePlay() {
    if (isPlaying) {
      player.pause()
    } else {
      player.play()
    }
  }

  function toggleMute() {
    player.emit('mutedChange', { muted: !player.muted, oldMuted: player.muted })
  }

  function skipForward() {
    player.seekBy(5)
  }

  function skipBackward() {
    player.seekBy(-5)
  }

  useEffect(() => {
    const timer = setTimeout(() => {
      if (showControls && isPlaying) {
        setShowControls(false)
      }
    }, 3000)

    return () => clearTimeout(timer)
  }, [showControls, isPlaying])

  function formatTime(seconds: number) {
    const minutes = Math.floor(seconds / 60)
    const remainingSeconds = Math.floor(seconds % 60)
    return `${minutes}:${remainingSeconds.toString().padStart(2, '0')}`
  }

  return (
    <View className={clsx(className, 'relative')}>
      <VideoView
        nativeControls={false}
        style={{ width, height }}
        player={player}
        allowsPictureInPicture
        fullscreenOptions={{ enable: true }}
      />
      <Pressable
        onPress={() => setShowControls(!showControls)}
        className={clsx('z-10 absolute inset-0 justify-between p-4', {
          'bg-black/50': showControls,
        })}
      >
        {showControls && (
          <>
            <View className="flex-1 justify-center items-center">
              <View className="flex-row items-center gap-5">
                <TouchableOpacity
                  onPress={skipBackward}
                  className="bg-black/70 rounded-full p-4"
                >
                  <Ionicons name="play-back" size={24} color="white" />
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={togglePlay}
                  className="bg-black/70 rounded-full p-5"
                >
                  <Ionicons
                    name={isPlaying ? 'pause' : 'play'}
                    size={32}
                    color="white"
                  />
                </TouchableOpacity>
                <TouchableOpacity
                  onPress={skipForward}
                  className="bg-black/70 rounded-full p-4"
                >
                  <Ionicons name="play-forward" size={24} color="white" />
                </TouchableOpacity>
              </View>
            </View>
            <View className="flex-row justify-between items-center">
              <Text className="text-white shrink">
                {formatTime(currentTime)} / {formatTime(duration)}
              </Text>
              <View className="flex-row gap-4">
                <TouchableOpacity onPress={toggleMute}>
                  <Ionicons
                    name={muted ? 'volume-mute' : 'volume-high'}
                    size={20}
                    color="white"
                  />
                </TouchableOpacity>
              </View>
            </View>
          </>
        )}
      </Pressable>
    </View>
  )
}
