type Alpha = number;
type Amount = number;
type ColorKeysTuple = [string, string, string];
type ColorModel = HSL | LAB | LCH | RGB;
type ColorModelKey = 'hsl' | 'oklab' | 'oklch' | 'rgb';
type ColorModelKeys<TModel extends ColorModelKey> = TModel extends 'hsl' ? keyof Omit<HSL, 'alpha'> : TModel extends 'oklab' ? keyof Omit<LAB, 'alpha'> : TModel extends 'oklch' ? keyof Omit<LCH, 'alpha'> : TModel extends 'rgb' ? keyof Omit<RGB, 'alpha'> : never;
type ColorReturn<T extends ColorType> = T extends 'hex' ? HEX : T extends 'hsl' ? HSL : T extends 'oklab' ? LAB : T extends 'oklch' ? LCH : T extends 'rgb' ? RGB : never;
type ColorTokens = 50 | 100 | 200 | 300 | 400 | 500 | 600 | 700 | 800 | 900 | 950;
type ColorTuple = [number, number, number];
type ColorType = 'hex' | 'hsl' | 'oklab' | 'oklch' | 'rgb';
type ConverterParameters<TModel extends ColorModel> = TModel | ColorTuple;
type Degrees = number;
type HEX = `#${string}`;
type PlainObject<T = any> = Record<string, T>;
interface Analysis {
    brightnessDifference: number;
    colorDifference: number;
    compliant: number;
    contrast: number;
    largeAA: boolean;
    largeAAA: boolean;
    normalAA: boolean;
    normalAAA: boolean;
}
interface Colors {
    alpha: Alpha;
    hex: HEX;
    hsl: HSL;
    oklab: LAB;
    oklch: LCH;
    rgb: RGB;
    type: ColorType;
}
interface HSL {
    alpha?: Alpha;
    h: number;
    l: number;
    s: number;
}
interface LAB {
    a: number;
    alpha?: Alpha;
    b: number;
    l: number;
}
interface LCH {
    alpha?: Alpha;
    c: number;
    h: number;
    l: number;
}
interface RGB {
    alpha?: Alpha;
    b: number;
    g: number;
    r: number;
}

interface ColorizrOptions {
    /**
     * Output color format.
     *
     * If not specified, the output will use the same format as the input color.
     */
    format?: ColorType;
}
declare class Colorizr {
    alpha: Alpha;
    hex: HEX;
    hsl: HSL;
    oklab: LAB;
    oklch: LCH;
    rgb: RGB;
    type: ColorType;
    constructor(color: string | HSL | LAB | LCH | RGB, options?: ColorizrOptions);
    /**
     * Get css string
     */
    get css(): string;
    /**
     * Get the red value
     */
    get red(): number;
    /**
     * Get the green value
     */
    get green(): number;
    /**
     * Get the blue value
     */
    get blue(): number;
    /**
     * Get the hue value
     */
    get hue(): number;
    /**
     * Get the saturation value
     */
    get saturation(): number;
    /**
     * Get the lightness value
     */
    get lightness(): number;
    /**
     * Get the luminance value
     */
    get luminance(): number;
    /**
     * Get the chroma value
     */
    get chroma(): number;
    get opacity(): number;
    /**
     * Get the contrasted color
     */
    get textColor(): string;
    private get selectedColor();
    brightnessDifference(input: string): number;
    colorDifference(input: string): number;
    /**
     * Test 2 colors for compliance
     */
    compare(input: string): Analysis;
    contrast(input: string): number;
    format(type: ColorType, precision?: number): string;
    /**
     * Increase lightness
     */
    lighten(amount: Amount): string;
    /**
     * Decrease lightness
     */
    darken(amount: Amount): string;
    /**
     * Increase saturation
     */
    saturate(amount: Amount): string;
    /**
     * Decrease saturation
     */
    desaturate(amount: Amount): string;
    /**
     * Invert color
     */
    invert(): string;
    /**
     * Add opacity to the color.
     */
    opacify(alpha?: Alpha): string;
    /**
     * Rotate color
     */
    rotate(degrees: Degrees): string;
    /**
     * Make the color more transparent
     */
    transparentize(alpha?: Alpha): string;
}

/**
 * Get the brightness difference between 2 colors.
 */
declare function brightnessDifference(left: string, right: string, precision?: number): number;

/**
 * Get the chroma of a color.
 */
declare function chroma(input: string): number;

/**
 * Get the difference between 2 colors.
 */
declare function colorDifference(left: string, right: string): number;

/**
 * Check 2 colors for WCAG compliance.
 */
declare function compare(left: string, right: string): Analysis;

/**
 * Get the color contrast between 2 colors.
 */
declare function contrast(left: string, right: string): number;

/**
 * Convert a color string to another format.
 */
declare function convert(input: string, format: ColorType): string;

/** Convert HEX to HSL */
declare function hex2hsl(input: string): HSL;

/** Convert HEX to oklab */
declare function hex2oklab(input: string, precision?: number): LAB;

/** Convert HEX to oklch */
declare function hex2oklch(input: string, precision?: number): LCH;

/** Convert HEX to RGB */
declare function hex2rgb(input: string): RGB;

/** Convert HSL to HEX */
declare function hsl2hex(input: ConverterParameters<HSL>): HEX;

/** Convert HSL to oklab */
declare function hsl2oklab(input: ConverterParameters<HSL>, precision?: number): LAB;

/** Convert HSL to oklch */
declare function hsl2oklch(input: ConverterParameters<HSL>, precision?: number): LCH;

/** Convert HSL to RGB */
declare function hsl2rgb(input: ConverterParameters<HSL>): RGB;

/** Convert oklab to HEX */
declare function oklab2hex(input: ConverterParameters<LAB>): HEX;

/** Convert oklab to HSL */
declare function oklab2hsl(input: ConverterParameters<LAB>): HSL;

/** Convert oklab to oklch */
declare function oklab2oklch(input: ConverterParameters<LAB>, precision?: number): LCH;

/** Convert oklab to RGB */
declare function oklab2rgb(input: ConverterParameters<LAB>, precision?: number): RGB;

/** Convert oklch to HEX */
declare function oklch2hex(input: ConverterParameters<LCH>): HEX;

/** Convert oklch to HSL */
declare function oklch2hsl(input: ConverterParameters<LCH>): HSL;

/** Convert oklch to oklab */
declare function oklch2oklab(input: ConverterParameters<LCH>, precision?: number): LAB;

/** Convert oklch to RGB */
declare function oklch2rgb(input: ConverterParameters<LCH>, precision?: number): RGB;

/** Convert RGB to HEX */
declare function rgb2hex(input: ConverterParameters<RGB>): HEX;

/** Convert RGB to HSL */
declare function rgb2hsl(input: ConverterParameters<RGB>): HSL;

/** Convert RGB to oklab */
declare function rgb2oklab(input: ConverterParameters<RGB>, precision?: number): LAB;

/** Convert RGB to oklch */
declare function rgb2oklch(input: ConverterParameters<RGB>, precision?: number): LCH;

/**
 * Decrease color lightness
 */
declare function darken(input: string, amount: Amount, format?: ColorType): string;

/**
 * Decrease color saturation
 */
declare function desaturate(input: string, amount: Amount, format?: ColorType): string;

type ExtractColorPartsReturn = {
    alpha?: number;
    model: ColorModelKey;
} & PlainObject<number>;
/**
 * Extract the color parts from a CSS color string.
 * Hex colors are not supported.
 */
declare function extractColorParts(input: string): ExtractColorPartsReturn;

interface FormatCSSOptions {
    /**
     * The alpha value of the color.
     */
    alpha?: Alpha;
    /**
     * Output color format.
     * @default 'hex'
     */
    format?: ColorType;
    /**
     * The number of digits of the output.
     * @default 5
     */
    precision?: number;
    /**
     * The separator between the values.
     *
     * oklab and oklch always use space as a separator.
     * @default ' '
     */
    separator?: string;
}
declare function formatCSS<T extends ColorModel | HEX>(input: T, options?: FormatCSSOptions): string;

declare function formatHex(input: string): HEX;

declare function isValidColor(input: string): boolean;

/**
 * Increase color lightness
 */
declare function lighten(input: string, amount: Amount, format?: ColorType): string;

/**
 * Get the luminance of a color.
 */
declare function luminance(input: string): number;

/**
 * Add an alpha value to a hex string
 */
declare function addAlphaToHex(input: string, alpha: Alpha): string;
/**
 * Convert an alpha value to a hex value.
 */
declare function convertAlphaToHex(input: Alpha): string;
/**
 * Extract the alpha value from a hex string
 */
declare function extractAlphaFromHex(input: string): number;
declare function hexadecimalToNumber(input: string): number;
/**
 * Remove the alpha value from a hex string
 */
declare function removeAlphaFromHex(input: string): string;

declare function isHex(input: any): input is HEX;
/**
 * Check if an object contains HSL values
 * The input must be an object with keys 'h', 's', and 'l'
 * with values between 0 and 360 for hue or 0 and 100 for the others.
 */
declare function isHSL(input: unknown): input is HSL;
/**
 * Check if an object contains LAB values
 * The input must be an object with keys 'l', 'a', and 'b' with values between -1 and 1.
 */
declare function isLAB(input: unknown): input is LAB;
/**
 * Check if an object contains LAB values
 * The input must be an object with keys 'l', 'c', and 'h' with values between 0 and 360.
 */
declare function isLCH(input: unknown): input is LCH;
/**
 * Check if an object contains RGB values.
 * The input must be an object with keys 'r', 'g', and 'b' with values between 0 and 255.
 */
declare function isRGB(input: unknown): input is RGB;

/**
 * Get the name of a color.
 * Returns the hex value if the color is not found.
 */
declare function name(input: string): string;

/**
 * Make the color transparent
 */
declare function opacify(input: string, alpha: Alpha, format?: ColorType): string;

declare function opacity(input: string): number;

/**
 * Get the maximum chroma for a given lightness and hue in the OkLCH color space
 */
declare function getOkLCHMaxChroma(input: string | LCH, precision?: number): number;
/**
 * Get a OkLCH color in the P3 color space.
 */
declare function getP3Color(input: string | LCH): string;

interface PaletteOptions {
    /**
     * Output color format.
     *
     * If not specified, the output will use the same format as the input color.
     */
    format?: ColorType;
    /**
     * Adjusts the lightness of the base color before generating the palette.
     *
     * Value should be between 0 and 100.
     */
    lightness?: number;
    /**
     * Adjusts the saturation of the base color before generating the palette.
     *
     * Value should be between 0 and 100.
     */
    saturation?: number;
    /**
     * The number of colors to generate in the palette.
     *
     * Minimum value is 2.
     * @default 6
     */
    size?: number;
    /**
     * Generate a monochromatic palette.
     *
     * For more options, use the `swatch` function.
     */
    type?: 'monochromatic';
}
declare function palette(input: string, options?: PaletteOptions): string[];

/**
 * Parse CSS color
 */
declare function parseCSS<T extends ColorType>(input: string, format?: T): ColorReturn<T>;

/**
 * Generate a random color.
 */
declare function random(type?: ColorType): string;

/**
 * Change the color hue
 */
declare function rotate(input: string, degrees: Degrees, format?: ColorType): string;

/**
 * Increase color saturation
 */
declare function saturate(input: string, amount: number, format?: ColorType): string;

type Scheme = 'analogous' | 'complementary' | 'rectangle' | 'split' | 'split-complementary' | 'square' | 'tetradic' | 'triadic';
interface SchemeOptions {
    /**
     * Output color format.
     *
     * If not specified, the output will use the same format as the input color.
     */
    format?: ColorType;
    /**
     * The type of scheme to generate.
     * @default 'complementary'
     */
    type?: Scheme;
}
/**
 * Get the scheme for a color.
 */
declare function scheme(input: string, typeOrOptions?: Scheme | SchemeOptions): string[];

type Swatch = {
    [key in ColorTokens]: string;
};
type SwatchVariant = 'deep' | 'neutral' | 'pastel' | 'subtle' | 'vibrant';
interface SwatchOptions {
    /**
     * Output color format.
     *
     * Determines the format of the generated colors (e.g., HEX, RGB, OKLCH, etc.).
     *
     * If not specified, the output will match the format of the input color.
     */
    format?: ColorType;
    /**
     * The lightness tuning factor for the swatch.
     * - 1: Linear lightness distribution.
     * - >1: Lighter tones are emphasized.
     * - <1: Darker tones are emphasized.
     * @default 1.5
     */
    lightnessFactor?: number;
    /**
     * The maximum lightness value for the swatch.
     *
     * Defines the upper bound for the lightest color in the palette.
     *
     * A number between 0 and 1.
     * @default 0.97
     */
    maxLightness?: number;
    /**
     * The minimum lightness value for the swatch.
     *
     * Defines the lower bound for the darkest color in the palette.
     *
     * A number between 0 and 1.
     *
     * @default 0.2
     */
    minLightness?: number;
    /**
     * Determines the scale type for the swatch.
     * - 'fixed': Shades are distributed with pre-defined lightness intervals.
     * - 'dynamic': Shades are distributed adaptively based on the input color.
     * @default 'dynamic'
     */
    scale?: 'dynamic' | 'fixed';
    /**
     * The variant of the swatch.
     * - 'deep': Generates rich and bold tones with significantly reduced lightness.
     * - 'neutral': Generates muted tones by reducing chroma.
     * - 'pastel': Produces soft and airy tones with significant chroma reduction.
     * - 'subtle': Creates extremely desaturated tones, close to grayscale.
     * - 'vibrant': Enhances chroma for bold and striking tones.
     */
    variant?: SwatchVariant;
}
/**
 * Generate a swatch of colors based on the input color
 *
 * This utility is ideal for designers and developers who need dynamic color
 * palettes for UI themes, design systems, or data visualization. Supports
 * multiple modes, scales, and variants for flexibility.
 */
declare function swatch(input: string, options?: SwatchOptions): Swatch;

interface TextColorOptions {
    /**
     * The dark color to return if the input is light.
     * @default '#000000'
     */
    darkColor?: string;
    /**
     * The light color to return if the input is dark.
     * @default '#ffffff'
     */
    lightColor?: string;
    /**
     * The threshold to determine if the color is light or dark.
     *
     * A number between 0 and 255.
     * @default 128
     */
    threshold?: number;
}
/**
 * Get the contrasted color for a given hex.
 */
declare function textColor(input: string, options?: TextColorOptions): string;

/**
 * Increase the color transparency.
 */
declare function transparentize(input: string, alpha: Alpha, format?: ColorType): string;

export { type Alpha, type Amount, type Analysis, type ColorKeysTuple, type ColorModel, type ColorModelKey, type ColorModelKeys, type ColorReturn, type ColorTokens, type ColorTuple, type ColorType, type ColorizrOptions, type Colors, type ConverterParameters, type Degrees, type FormatCSSOptions, type HEX, type HSL, type LAB, type LCH, type PaletteOptions, type PlainObject, type RGB, type Scheme, type SchemeOptions, type Swatch, type SwatchOptions, type SwatchVariant, type TextColorOptions, addAlphaToHex, brightnessDifference, chroma, colorDifference, compare, contrast, convert, convertAlphaToHex, darken, Colorizr as default, desaturate, extractAlphaFromHex, extractColorParts, formatCSS, formatHex, getOkLCHMaxChroma, getP3Color, hex2hsl, hex2oklab, hex2oklch, hex2rgb, hexadecimalToNumber, hsl2hex, hsl2oklab, hsl2oklch, hsl2rgb, isHSL, isHex, isLAB, isLCH, isRGB, isValidColor, lighten, luminance, name, oklab2hex, oklab2hsl, oklab2oklch, oklab2rgb, oklch2hex, oklch2hsl, oklch2oklab, oklch2rgb, opacify, opacity, palette, parseCSS, random, removeAlphaFromHex, rgb2hex, rgb2hsl, rgb2oklab, rgb2oklch, rotate, saturate, scheme, swatch, textColor, transparentize };
