'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

var solidJs = require('solid-js');

/* @jsxImportSource solid-js */
function defaultCompare(a, b) {
  return a === b;
}
function useSelectorWithCompare(selector, compare) {
  let previous;
  return state => {
    const next = selector(state);
    if (previous === undefined || !compare(previous, next)) {
      previous = next;
    }
    return previous;
  };
}

/**
 * Creates a selector which subscribes to the store and selects a value from the
 * store's snapshot, using an optional comparison function.
 *
 * @example
 *
 * ```tsx
 * import { donutStore } from './donutStore.ts';
 * import { useSelector } from '@xstate/store/solid';
 *
 * function DonutCounter() {
 *   const donutCount = useSelector(donutStore, (state) => state.context.donuts);
 *
 *   return (
 *     <div>
 *       <button onClick={() => donutStore.send({ type: 'addDonut' })}>
 *         Add donut ({donutCount()})
 *       </button>
 *     </div>
 *   );
 * }
 * ```
 *
 * @param store The store, created from `createStore(…)`
 * @param selector A function which takes in the snapshot and returns a selected
 *   value from it
 * @param compare An optional function which compares the selected value to the
 *   previously selected value
 * @returns A read-only signal of the selected value
 */
function useSelector(store, selector, compare = defaultCompare) {
  const selectorWithCompare = useSelectorWithCompare(selector, compare);
  const [selectedValue, setSelectedValue] = solidJs.createSignal(selectorWithCompare(store.getSnapshot()));
  solidJs.createEffect(() => {
    const subscription = store.subscribe(() => {
      const newValue = selectorWithCompare(store.getSnapshot());
      setSelectedValue(() => newValue);
    });
    solidJs.onCleanup(() => {
      subscription.unsubscribe();
    });
  });
  return selectedValue;
}

exports.useSelector = useSelector;
