import SwiftUI
import KotlinSharedUI

enum Route: Hashable, Identifiable {
    var id: Int {
        return self.hashValue
    }
    
    static func == (lhs: Route, rhs: Route) -> Bool {
        return lhs.hashValue == rhs.hashValue
    }

    @MainActor
    @ViewBuilder
    func view(
        onNavigate: @escaping (Route) -> Void,
        clearToHome: @escaping () -> Void
    ) -> some View {
        switch self {
        case .home(let accountType): HomeTimelineScreen(accountType: accountType, toServiceSelect: { onNavigate(.serviceSelect) }, toCompose: { onNavigate(.composeNew(accountType)) }, toTabSetting: { onNavigate(.tabSettings) })
        case .timeline(let item):
            switch onEnum(of: item) {
            case .ListTimelineTabItem(let listTabItem):
                ListTimelineScreen(tabItem: listTabItem)
                    .navigationTitle(item.metaData.title.text)
            default:
                TimelineScreen(tabItem: item)
                    .navigationTitle(item.metaData.title.text)
            }
        case .serviceSelect:
            ServiceSelectionScreen(toHome: { clearToHome() })
        case .statusDetail(let accountType, let statusKey):
            StatusDetailScreen(accountType: accountType, statusKey: statusKey)
        case .profileUser(let accountType, let userKey):
            ProfileScreen(accountType: accountType, userKey: userKey, onFollowingClick: { key in onNavigate(.userFollowing(accountType, key)) }, onFansClick: { key in onNavigate(.userFans(accountType, key)) })
        case .settings:
            SettingsScreen()
        case .tabItem(let tabItem):
            tabItem.view(onNavigate: onNavigate)
        case .accountManagement:
            AccountManagementScreen()
        case .search(let accountType, let query):
            SearchScreen(accountType: accountType, initialQuery: query)
        case .composeNew(let accountType):
            ComposeScreen(accountType: accountType)
        case .composeQuote(let accountType, let statusKey):
            ComposeScreen(accountType: accountType, composeStatus: ComposeStatus.Quote(statusKey: statusKey))
        case .composeReply(let accountType, let statusKey):
            ComposeScreen(accountType: accountType, composeStatus: ComposeStatus.Reply(statusKey: statusKey))
        case .composeVVOReplyComment(let accountType, let statueKey, let rootId):
            ComposeScreen(accountType: accountType, composeStatus: ComposeStatus.VVOComment(statusKey: statueKey, rootId: rootId))
        case .profileUserNameWithHost(let accountType, let userName, let host):
            ProfileWithUserNameAndHostScreen(userName: userName, host: host, accountType: accountType, onFollowingClick: { key in onNavigate(.userFollowing(accountType, key)) }, onFansClick: { key in onNavigate(.userFans(accountType, key)) })
        case .appearance:
            AppearanceScreen()
        case .about:
            AboutScreen()
        case .localHostory:
            LocalHistoryScreen()
        case .storage:
            StorageScreen()
        case .localFilter:
            LocalFilterScreen()
        case .aiConfig:
            AiConfigScreen()
        case .tabSettings:
            TabSettingsScreen()
        case .rssDetail(let url):
            RssDetailScreen(url: url)
        case .statusVVOStatus(let accountType, let statusKey):
            VVOStatusScreen(accountType: accountType, statusKey: statusKey)
        case .statusVVOComment(let accountType, let statusKey):
            VVOCommentScreen(accountType: accountType, statusKey: statusKey)
        case .statusBlueskyReport(let accountType, let statusKey):
            BlueskyReportSheet(accountType: accountType, statusKey: statusKey)
        case .statusMisskeyReport(let accountType, let userKey, let statusKey):
            MisskeyReportSheet(accountType: accountType, userKey: userKey, statusKey: statusKey)
        case .statusAddReaction(let accountType, let statusKey):
            StatusAddReactionSheet(accountType: accountType, statusKey: statusKey)
        case .userFans(let account, let userKey):
            UserListScreen(accountType: account, userKey: userKey, isFollowing: false)
        case .userFollowing(let account, let userKey):
            UserListScreen(accountType: account, userKey: userKey, isFollowing: true)
        case .dmConversation(let accountType, let roomKey, let title):
            DMConversationScreen(accountType: accountType, roomKey: roomKey)
                .navigationTitle(title)
        case .mediaImage(let url, let preview):
            MediaScreen(url: url)
        case .mediaStatusMedia(let accountType, let statusKey, let selectedIndex, let preview):
            StatusMediaScreen(accountType: accountType, statusKey: statusKey, initialIndex: Int(selectedIndex), preview: preview)
        case .appLog:
            AppLogScreen()
        case .deepLinkAccountPicker(let originalUrl, let data):
            DeepLinkAccountPicker(originalUrl: originalUrl, data: data, onNavigate: onNavigate)
        default:
            Text("Not done yet for \(self)")
        }
    }

    case home(AccountType)
    case timeline(TimelineTabItem)
    case composeNew(AccountType)
    case composeQuote(AccountType, MicroBlogKey)
    case composeReply(AccountType, MicroBlogKey)
    case composeVVOReplyComment(AccountType, MicroBlogKey, String)
    case mediaImage(String, String?)
    case mediaPodcast(AccountType, String)
    case mediaStatusMedia(AccountType, MicroBlogKey, Int32, String?)
    case profileUser(AccountType, MicroBlogKey)
    case profileUserNameWithHost(AccountType, String, String)
    case rssDetail(String)
    case search(AccountType, String)
    case statusAddReaction(AccountType, MicroBlogKey)
    case statusAltText(String)
    case statusBlueskyReport(AccountType, MicroBlogKey)
    case statusDeleteConfirm(AccountType, MicroBlogKey)
    case statusDetail(AccountType, MicroBlogKey)
    case statusMastodonReport(AccountType, MicroBlogKey, MicroBlogKey?)
    case statusMisskeyReport(AccountType, MicroBlogKey, MicroBlogKey?)
    case statusVVOComment(AccountType, MicroBlogKey)
    case statusVVOStatus(AccountType, MicroBlogKey)
    case serviceSelect
    case accountManagement
    case localFilter
    case localHostory
    case moreMenuCustomize
    case aiConfig
    case storage
    case appearance
    case settings
    case about
    case tabItem(TabItem)
    case tabSettings
    case userFollowing(AccountType, MicroBlogKey)
    case userFans(AccountType, MicroBlogKey)
    case dmConversation(AccountType, MicroBlogKey, String)
    case appLog
    case deepLinkAccountPicker(String, [MicroBlogKey: Route])

    fileprivate static func fromCompose(_ compose: DeeplinkRoute.Compose) -> Route? {
        switch onEnum(of: compose) {
        case .new(let data):
            return Route.composeNew(data.accountType)
        case .quote(let data):
            return Route.composeQuote(AccountType.Specific(accountKey: data.accountKey), data.statusKey)
        case .reply(let data):
            return Route.composeReply(AccountType.Specific(accountKey: data.accountKey), data.statusKey)
        case .vVOReplyComment(let data):
            return Route.composeVVOReplyComment(AccountType.Specific(accountKey: data.accountKey), data.replyTo, data.rootId)
        }
    }
    
    fileprivate static func fromMedia(_ media: DeeplinkRoute.Media) -> Route? {
        switch onEnum(of: media) {
        case .image(let data):
            return Route.mediaImage(data.uri, data.previewUrl)
        case .podcast(let data):
            return Route.mediaPodcast(data.accountType, data.id)
        case .statusMedia(let data):
            return Route.mediaStatusMedia(data.accountType, data.statusKey, Int32(data.index), data.preview)
        }
    }
    
    fileprivate static func fromProfile(_ profile: DeeplinkRoute.Profile) -> Route? {
        switch onEnum(of: profile) {
        case .user(let data):
            return Route.profileUser(data.accountType, data.userKey)
        case .userNameWithHost(let data):
            return Route.profileUserNameWithHost(data.accountType, data.userName, data.host)
        }
    }
    
    fileprivate static func fromStatus(_ status: DeeplinkRoute.Status) -> Route? {
        switch onEnum(of: status) {
        case .addReaction(let data):
            return Route.statusAddReaction(data.accountType, data.statusKey)
        case .altText(let data):
            return Route.statusAltText(data.text)
        case .blueskyReport(let data):
            return Route.statusBlueskyReport(data.accountType, data.statusKey)
        case .deleteConfirm(let data):
            return Route.statusDeleteConfirm(data.accountType, data.statusKey)
        case .detail(let data):
            return Route.statusDetail(data.accountType, data.statusKey)
        case .mastodonReport(let data):
            return Route.statusMastodonReport(data.accountType, data.userKey, data.statusKey)
        case .misskeyReport(let data):
            return Route.statusMisskeyReport(data.accountType, data.userKey, data.statusKey)
        case .vVOComment(let data):
            return Route.statusVVOComment(data.accountType, data.commentKey)
        case .vVOStatus(let data):
            return Route.statusVVOStatus(data.accountType, data.statusKey)
        }
    }
    
    static func fromDeepLink(url: String) -> Route? {
        if let deeplinkRoute = DeeplinkRoute.companion.parse(url: url) {
            return fromDeepLinkRoute(deeplinkRoute: deeplinkRoute)
        } else {
            return nil
        }
    }
    
    static func fromDeepLinkRoute(deeplinkRoute: DeeplinkRoute) -> Route? {
        switch onEnum(of: deeplinkRoute) {
        case .compose(let compose):
            return fromCompose(compose)
        case .media(let media):
            return fromMedia(media)
        case .profile(let profile):
            return fromProfile(profile)
        case .rss(let rss):
            switch onEnum(of: rss) {
            case .detail(let data):
                return Route.rssDetail(data.url)
            }
        case .search(let search):
            return Route.search(search.accountType, search.query)
        case .status(let status):
            return fromStatus(status)
        case .login:
            return Route.serviceSelect
        case .deepLinkAccountPicker(let picker):
            return .deepLinkAccountPicker(picker.originalUrl, picker.data.mapValues { route in
                return fromDeepLinkRoute(deeplinkRoute: route)
            }.compactMapValues { $0 })
        case .openLinkDirectly(_):
            return nil
        }
    }
}
