package dev.dimension.flare.ui.model

import androidx.compose.runtime.Immutable
import dev.dimension.flare.common.AppDeepLink
import dev.dimension.flare.data.datasource.microblog.StatusAction
import dev.dimension.flare.model.MicroBlogKey
import dev.dimension.flare.model.PlatformType
import dev.dimension.flare.ui.humanizer.humanize
import dev.dimension.flare.ui.model.mapper.MisskeyAchievement
import dev.dimension.flare.ui.render.UiDateTime
import dev.dimension.flare.ui.render.UiRichText
import kotlinx.collections.immutable.ImmutableList
import kotlinx.collections.immutable.ImmutableMap
import kotlinx.collections.immutable.persistentListOf

@Immutable
public data class UiTimeline internal constructor(
    val topMessage: TopMessage?,
    val content: ItemContent?,
    private val dbKey: String? = null,
) {
    val itemKey: String
        get() =
            buildString {
                if (topMessage != null) {
                    append("withTopMessage")
                    append(topMessage.itemKey)
                }
                if (content != null) {
                    append("withContent")
                    append(content.itemKey)
                }
                if (dbKey != null) {
                    append("withDbKey")
                    append(dbKey)
                }
            }
    val itemType: String
        get() =
            buildString {
//                append(platformType.name)
                if (topMessage != null) {
                    append("withTopMessage")
                }
                if (content != null) {
                    append("withContent")
                    when (content) {
                        is ItemContent.Status -> {
                            append("Status")
                        }
                        is ItemContent.User -> {
                            append("User")
                        }
                        is ItemContent.UserList -> {
                            append("UserList")
                        }

                        is ItemContent.Feed -> {
                            append("Feed")
                        }
                    }
                }
            }

    public sealed class ItemContent {
        public abstract val itemKey: String

        public data class Feed internal constructor(
            val title: String,
            val description: String?,
            val url: String,
            val image: String?,
            val createdAt: UiDateTime?,
            val source: String,
            val sourceIcon: String?,
            val imageHeaders: ImmutableMap<String, String>?,
            private val openInBrowser: Boolean,
        ) : ItemContent() {
            override val itemKey: String
                get() = "Feed_$url"

            val onClicked: ClickContext.() -> Unit = {
                if (openInBrowser) {
                    launcher.launch(url)
                } else {
                    launcher.launch(AppDeepLink.RSS.invoke(url))
                }
            }
        }

        public data class Status internal constructor(
            val platformType: PlatformType,
            val images: ImmutableList<UiMedia>,
            val sensitive: Boolean,
            val contentWarning: UiRichText?,
            val user: UiUserV2?,
            val quote: ImmutableList<Status>,
            val content: UiRichText,
            val actions: ImmutableList<StatusAction>,
            val poll: UiPoll?,
            val statusKey: MicroBlogKey,
            val card: UiCard?,
            val createdAt: UiDateTime,
            val bottomContent: BottomContent? = null,
            val topEndContent: TopEndContent? = null,
            val aboveTextContent: AboveTextContent? = null,
            val parents: ImmutableList<Status> = persistentListOf(),
            val url: String,
            val onClicked: ClickContext.() -> Unit,
            val onMediaClicked: ClickContext.(media: UiMedia, index: Int) -> Unit,
        ) : ItemContent() {
            override val itemKey: String
                get() =
                    buildString {
                        append(platformType.name)
                        append("Status")
                        append(statusKey)
                    }

            val shouldExpandTextByDefault: Boolean by lazy {
                contentWarning == null && !content.isLongText
            }

            public sealed class BottomContent {
                public data class Reaction internal constructor(
                    val emojiReactions: ImmutableList<EmojiReaction>,
                ) : BottomContent() {
                    public data class EmojiReaction internal constructor(
                        val name: String,
                        val url: String,
                        val count: Long,
                        val onClicked: () -> Unit,
                        // TODO: make EmojiReaction a sealed class
                        val isUnicode: Boolean,
                        val me: Boolean,
                    ) {
                        val humanizedCount: String by lazy {
                            count.humanize()
                        }
                        val isImageReaction: Boolean by lazy {
                            name.startsWith(":") && name.endsWith(":")
                        }
                    }
                }
            }

            public sealed class TopEndContent {
                public data class Visibility internal constructor(
                    val visibility: Type,
                ) : TopEndContent() {
                    public enum class Type {
                        Public,
                        Home,
                        Followers,
                        Specified,
                    }
                }
            }

            public sealed class AboveTextContent {
                public data class ReplyTo internal constructor(
                    val handle: String,
                ) : AboveTextContent()
            }
        }

        public data class User internal constructor(
            val value: UiUserV2,
            val button: ImmutableList<Button> = persistentListOf(),
        ) : ItemContent() {
            override val itemKey: String
                get() =
                    buildString {
                        append("User")
                        append(value.key)
                    }

            public sealed class Button {
                public data class AcceptFollowRequest internal constructor(
                    val onClicked: ClickContext.() -> Unit,
                ) : Button()

                public data class RejectFollowRequest internal constructor(
                    val onClicked: ClickContext.() -> Unit,
                ) : Button()
            }
        }

        public data class UserList internal constructor(
            val users: ImmutableList<UiUserV2>,
            val status: Status? = null,
        ) : ItemContent() {
            override val itemKey: String
                get() =
                    buildString {
                        append("UserList")
                        append(users.hashCode())
                        if (status != null) {
                            append(status.itemKey)
                        }
                    }
        }
    }

    public data class TopMessage internal constructor(
        val user: UiUserV2?,
        val icon: Icon,
        val type: MessageType,
        val onClicked: ClickContext.() -> Unit,
        val statusKey: MicroBlogKey,
    ) {
        val itemKey: String
            get() =
                buildString {
                    append("TopMessage")
                    append(type)
                    if (user != null) {
                        append(user.key)
                    }
                    append(statusKey.toString())
                }

        public enum class Icon {
            Retweet,
            Follow,
            Favourite,
            Mention,
            Poll,
            Edit,
            Info,
            Reply,
            Quote,
            Pin,
        }

        public sealed class MessageType {
            public sealed class Mastodon : MessageType() {
                public data class Reblogged internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Follow internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Favourite internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Mention internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Poll internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class FollowRequest internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Status internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Update internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class UnKnown internal constructor(
                    val id: String,
                ) : Mastodon()

                public data class Pinned internal constructor(
                    val id: String,
                ) : Mastodon()
            }

            public sealed class Misskey : MessageType() {
                public data class Follow internal constructor(
                    val id: String,
                ) : Misskey()

                public data class Mention internal constructor(
                    val id: String,
                ) : Misskey()

                public data class Reply internal constructor(
                    val id: String,
                ) : Misskey()

                public data class Renote internal constructor(
                    val id: String,
                ) : Misskey()

                public data class Quote internal constructor(
                    val id: String,
                ) : Misskey()

                public data class Reaction internal constructor(
                    val id: String,
                ) : Misskey()

                public data class PollEnded internal constructor(
                    val id: String,
                ) : Misskey()

                public data class ReceiveFollowRequest internal constructor(
                    val id: String,
                ) : Misskey()

                public data class FollowRequestAccepted internal constructor(
                    val id: String,
                ) : Misskey()

                public data class AchievementEarned internal constructor(
                    val id: String,
                    val achievement: MisskeyAchievement?,
                ) : Misskey()

                public data class App internal constructor(
                    val id: String,
                ) : Misskey()

                public data class UnKnown internal constructor(
                    val type: String,
                    val id: String,
                ) : Misskey()

                public data class Pinned internal constructor(
                    val id: String,
                ) : Misskey()
            }

            public sealed class Bluesky : MessageType() {
                public data object Like : Bluesky()

                public data object Repost : Bluesky()

                public data object Follow : Bluesky()

                public data object Mention : Bluesky()

                public data object Reply : Bluesky()

                public data object Quote : Bluesky()

                public data object UnKnown : Bluesky()

                public data object StarterpackJoined : Bluesky()

                public data object Pinned : Bluesky()
            }

            public sealed class XQT : MessageType() {
                public data object Retweet : XQT()

                public data class Custom internal constructor(
                    val message: String,
                    val id: String,
                ) : XQT() {
                    override fun toString(): String = "Custom$id"
                }

                public data object Mention : XQT()
            }

            public sealed class VVO : MessageType() {
                public data class Custom internal constructor(
                    val message: String,
                ) : VVO()

                public data object Like : VVO()
            }
        }
    }
}
