import SwiftUI
import KotlinSharedUI
import SwiftUIBackports

struct StatusActionsView: View {
    let data: [StatusAction]
    let useText: Bool

    var body: some View {
        HStack {
            ForEach(0..<data.count, id: \.self) { index in
                let item = data[index]
                StatusActionView(data: item, useText: useText, isFixedWidth: index != data.count - 1)
                    .if(index == data.count - 1) { view in
                        view.frame(maxWidth: .infinity, alignment: .trailing)
                    } else: { view in
                        view
                    }

            }
        }
        .backport
        .labelIconToTitleSpacing(4)
    }
}

struct StatusActionView: View {
    @Environment(\.themeSettings) private var themeSettings
    let data: StatusAction
    let useText: Bool
    let isFixedWidth: Bool
    var body: some View {
        switch onEnum(of: data) {
        case .item(let item):
            StatusActionItemView(data: item, useText: useText, isFixedWidth: isFixedWidth)
        case .group(let group):
            if useText {
                Divider()
                ForEach(0..<group.actions.count, id: \.self) { index in
                    let item = group.actions[index]
                    StatusActionView(data: item, useText: true, isFixedWidth: false)
                }
                Divider()
            } else {
                Menu {
                    ForEach(0..<group.actions.count, id: \.self) { index in
                        let item = group.actions[index]
                        StatusActionView(data: item, useText: true, isFixedWidth: false)
                    }
                } label: {
                    if !isFixedWidth && group.displayItem.countText == nil {
                        if let color = group.displayItem.color {
                            StatusActionIcon(item: group.displayItem)
                                .foregroundStyle(color)
                                .scaledToFit()
                                .frame(width: 32, alignment: .center)
                                .contentShape(.rect)
                        } else {
                            StatusActionIcon(item: group.displayItem)
                                .scaledToFit()
                                .frame(width: 32, alignment: .center)
                                .contentShape(.rect)
                        }
                    } else {
                        Label {
                            ZStack(
                                alignment: .leading
                            ) {
                                if isFixedWidth, !useText {
                                    Text("0000")
                                        .opacity(0.0)
                                }
                                if let text = group.displayItem.countText, themeSettings.appearanceSettings.showNumbers {
                                    if let color = group.displayItem.color {
                                        Text(text)
                                            .foregroundStyle(color)
                                    } else {
                                        Text(text)
                                    }
                                }
                            }
                        } icon: {
                            if let color = group.displayItem.color {
                                StatusActionIcon(item: group.displayItem)
                                    .foregroundStyle(color)
                            } else {
                                StatusActionIcon(item: group.displayItem)
                            }
                        }
                    }
                }
            }
        case .asyncActionItem(let asyncItem):
            AsyncStatusActionView(data: asyncItem)
        }
    }
}

struct AsyncStatusActionView: View {
    let data: StatusActionAsyncActionItem
    var body: some View {
        Button {

        } label: {

        }
        .onAppear {
        }
    }
}

struct StatusActionItemView: View {
    @Environment(\.themeSettings) private var themeSettings
    @Environment(\.openURL) private var openURL
    let data: StatusActionItem
    let useText: Bool
    let isFixedWidth: Bool
    var body: some View {
        if let shareable = data as? StatusActionItemShareable {
            ShareLink(data.textKey, item: .init(string: shareable.content)!)
        } else {
            Button(
                role: data.role,
            ) {
                if let clickable = data as? StatusActionItemClickable {
                    clickable.onClicked(ClickContext(launcher: AppleUriLauncher(openUrl: openURL)))
                }
            } label: {
                Label {
                    ZStack(
                        alignment: .leading
                    ) {
                        if isFixedWidth, !useText {
                            Text("0000")
                                .opacity(0.0)
                        }
                        if useText {
                            if let color = data.color {
                                Text(data.textKey)
                                    .foregroundStyle(color)
                            } else {
                                Text(data.textKey)
                            }
                        } else if let text = data.countText, themeSettings.appearanceSettings.showNumbers {
                            if let color = data.color {
                                Text(text)
                                    .foregroundStyle(color)
                            } else {
                                Text(text)
                            }
                        }
                    }
                } icon: {
                    if let color = data.color {
                        StatusActionIcon(item: data)
                            .foregroundStyle(color)
                    } else {
                        StatusActionIcon(item: data)
                    }
                }
            }
            .sensoryFeedback(.success, trigger: data.color)
            .buttonStyle(.plain)
        }
    }
}

extension StatusActionItem {
    var countText: String? {
        if let numbered = self as? StatusActionItemNumbered {
            return numbered.humanizedCount
        } else {
            return nil
        }
    }
    var color: Color? {
        if let colorized = self as? StatusActionItemColorized {
            switch colorized.color {
            case .contentColor: nil
            case .error: Color(.systemRed)
            case .primaryColor: Color.accentColor
            case .red: Color(.systemRed)
            }
        } else {
            nil
        }
    }
    var role: ButtonRole? {
        if let colorized = self as? StatusActionItemColorized {
            switch colorized.color {
            case .red:
                    .destructive
            case .error:
                    .destructive
            case .contentColor:
                    nil
            case .primaryColor:
                if #available(iOS 26.0, *) {
                    .confirm
                } else {
                    nil
                }
            }
        } else {
            nil
        }
    }

    var textKey: LocalizedStringResource {
        switch onEnum(of: self) {
        case .bookmark(let bookmarked):
            return bookmarked.bookmarked
                ? LocalizedStringResource("bookmark_remove")
                : LocalizedStringResource("bookmark_add")

        case .delete:
            return LocalizedStringResource("delete")

        case .like(let liked):
            return liked.liked
                ? LocalizedStringResource("unlike")
                : LocalizedStringResource("like")

        case .more:
            return LocalizedStringResource("more")

        case .quote:
            return LocalizedStringResource("quote")

        case .reaction(let reacted):
            return reacted.reacted
                ? LocalizedStringResource("reaction_remove")
                : LocalizedStringResource("reaction_add")

        case .reply:
            return LocalizedStringResource("reply")

        case .report:
            return LocalizedStringResource("report")

        case .retweet(let retweeted):
            return retweeted.retweeted
                ? LocalizedStringResource("retweet_remove")
                : LocalizedStringResource("retweet")
        case .comment:
            return LocalizedStringResource("comment")
        case .share:
            return LocalizedStringResource("share")
        case .fxShare:
            return LocalizedStringResource("fx_share")
        }
    }
}

struct StatusActionIcon: View {
    let item: StatusActionItem

    var body: some View {
        Group {
            switch onEnum(of: item) {
            case .bookmark(let bookmarked):
                if bookmarked.bookmarked {
                    Image("fa-bookmark.fill")
                } else {
                    Image("fa-bookmark")
                }

            case .delete:
                Image("fa-trash")

            case .like(let liked):
                if liked.liked {
                    Image("fa-heart.fill")
                } else {
                    Image("fa-heart")
                }

            case .more:
                Image("fa-ellipsis")

            case .quote:
                Image("fa-quote-left")

            case .reaction(let reacted):
                if reacted.reacted {
                    Image("fa-minus")
                } else {
                    Image("fa-plus")
                }

            case .reply:
                Image("fa-reply")

            case .report:
                Image("fa-circle-info")

            case .retweet:
                Image("fa-retweet")

            case .comment:
                Image("fa-comment-dots")
            case .share:
                Image(.faShareNodes)
            case .fxShare:
                Image(.faShareNodes)
            }
        }
    }
}
