package dev.develsinthedetails.eatpoopyoucat.ui.previousgames

import android.content.res.Configuration
import android.net.Uri
import androidx.activity.compose.ManagedActivityResultLauncher
import androidx.compose.foundation.background
import androidx.compose.foundation.border
import androidx.compose.foundation.clickable
import androidx.compose.foundation.gestures.detectTapGestures
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentWidth
import androidx.compose.foundation.lazy.grid.GridCells
import androidx.compose.foundation.lazy.grid.LazyVerticalGrid
import androidx.compose.foundation.lazy.grid.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.rounded.DeleteForever
import androidx.compose.material3.Card
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Surface
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.livedata.observeAsState
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.dimensionResource
import androidx.compose.ui.res.pluralStringResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.constraintlayout.compose.ConstraintLayout
import androidx.hilt.navigation.compose.hiltViewModel
import dev.develsinthedetails.eatpoopyoucat.R
import dev.develsinthedetails.eatpoopyoucat.data.GameWithEntries
import dev.develsinthedetails.eatpoopyoucat.data.entriesAreValid
import dev.develsinthedetails.eatpoopyoucat.ui.draw.DrawBox
import dev.develsinthedetails.eatpoopyoucat.ui.helpers.ConfirmDialog
import dev.develsinthedetails.eatpoopyoucat.ui.helpers.Scaffolds
import dev.develsinthedetails.eatpoopyoucat.ui.helpers.SpinnerScreen
import dev.develsinthedetails.eatpoopyoucat.ui.theme.AppTheme
import dev.develsinthedetails.eatpoopyoucat.viewmodels.PreviousGamesViewModel
import kotlinx.serialization.json.Json

@Composable
fun PreviousGamesScreen(
    modifier: Modifier = Modifier,
    viewModel: PreviousGamesViewModel = hiltViewModel(),
    onGoHome: () -> Unit = {},
    onGameClick: (String) -> Unit,
    onBackupGames: (games: List<GameWithEntries>?) -> Unit,
    onImportGames: ManagedActivityResultLauncher<String, Uri?>,
) {
    val games by viewModel.games.observeAsState(initial = null)
    if (games == null) {
        SpinnerScreen()
    } else {
        LaunchedEffect(key1 = games) {
            // clean up games that were created and never played.
            // These are created by the side effect of creating the game before the first sentence
            val invalidGames = games!!.filter { !it.entriesAreValid() }
            if(invalidGames.isNotEmpty())
                viewModel.cleanup(invalidGames)
        }

        PreviousGamesScreen(
            modifier,
            games = games!!.filter { it.entriesAreValid() },
            onBackupGames = { onBackupGames(games) },
            onImportGames = onImportGames,
            onGotoGame = onGameClick,
            onGoHome = onGoHome,
            onDelete = { viewModel.deleteGame(it) })
    }
}

@Composable
fun PreviousGamesScreen(
    modifier: Modifier = Modifier,
    games: List<GameWithEntries>,
    onGoHome: () -> Unit,
    onBackupGames: () -> Unit,
    onImportGames: ManagedActivityResultLauncher<String, Uri?>?,
    onGotoGame: (String) -> Unit,
    onDelete: (String) -> Unit,
) {
    Scaffolds.PreviousGames(
        title = pluralStringResource(
            id = R.plurals.previous_games, games.count()
        ),
        onBackupGames = onBackupGames,
        onImportGames = onImportGames,
        onBack = onGoHome
    )
    { innerPadding ->
        Surface(
            modifier = modifier
                .fillMaxSize()
                .padding(innerPadding),
            color = MaterialTheme.colorScheme.primaryContainer
        ) {
            if (games.isEmpty()) {
                Row(
                    modifier = Modifier
                        .fillMaxSize()
                        .clickable(
                            onClick = {
                                onGoHome()
                            }
                        ),
                    verticalAlignment = Alignment.CenterVertically
                ) {
                    Text(
                        modifier = Modifier
                            .padding(8.dp),
                        textAlign = TextAlign.Center,
                        fontSize = 30.sp,
                        lineHeight = 35.sp,
                        fontWeight = FontWeight.Bold,
                        text = stringResource(id = R.string.no_previous_games_found),
                    )
                }
            } else {
                LazyVerticalGrid(
                    columns = GridCells.Fixed(2),
                    modifier = modifier.testTag("game_list"),
                    contentPadding = PaddingValues(
                        horizontal = dimensionResource(id = R.dimen.card_side_margin),
                        vertical = dimensionResource(id = R.dimen.header_margin)
                    )
                ) {
                    items(
                        items = games,
                        key = { it.game.id }
                    ) { game ->
                        GameListItem(game = game,
                            onGotoGame = { onGotoGame(game.game.id.toString()) },
                            onDelete = { onDelete(game.game.id.toString()) })
                    }
                }
            }
        }
    }
}

@Composable
fun GameListItem(game: GameWithEntries, onGotoGame: () -> Unit, onDelete: () -> Unit) {
    val firstSentence = game.entries
        .minBy { it.sequence }.sentence ?: String()
    val lastDrawing = game.entries
        .sortedBy { it.sequence }
        .lastOrNull { it.drawing != null }
    ConstraintLayout(
        modifier = Modifier.fillMaxSize()
    ) {
        ListItem(
            sentence = firstSentence,
            drawing = lastDrawing?.drawing,
            turns = game.entries.count(),
            onGotoGame = onGotoGame
        )
        val deleteGame = createRef()
        var showDeleteConfirmation by remember { mutableStateOf(false) }
        IconButton(modifier = Modifier
            .background(
                color = MaterialTheme.colorScheme.tertiaryContainer,
                shape = RoundedCornerShape(32.dp)
            )
            .constrainAs(deleteGame)
            {
                end.linkTo(parent.end)
                bottom.linkTo(parent.bottom)
            },
            onClick = { showDeleteConfirmation = true }) {
            Icon(
                Icons.Rounded.DeleteForever,
                modifier = Modifier,
                contentDescription = stringResource(id = R.string.delete_game),
            )
        }
        if (showDeleteConfirmation) {
            ConfirmDialog(
                action = stringResource(id = R.string.delete_game),
                onDismiss = { showDeleteConfirmation = false },
                onConfirm = {
                    onDelete()
                    showDeleteConfirmation = false
                })
        }
    }
}

@Composable
fun ListItem(
    sentence: String,
    drawing: ByteArray?,
    turns: Int,
    onGotoGame: () -> Unit
) {
    Card(
        shape = MaterialTheme.shapes.small,
        modifier = Modifier
            .padding(horizontal = dimensionResource(id = R.dimen.card_side_margin))
            .padding(bottom = dimensionResource(id = R.dimen.card_bottom_margin))
            .pointerInput(Unit) {
                detectTapGestures(
                    onTap = { onGotoGame() }
                )
            }
    ) {
        Column(
            Modifier
                .fillMaxWidth()
                .padding(all = dimensionResource(id = R.dimen.margin_normal))
        ) {
            Text(
                text = sentence,
                textAlign = TextAlign.Start,
                maxLines = 2,
                modifier = Modifier
                    .fillMaxWidth()
                    .wrapContentWidth(Alignment.Start)
            )
            Text(
                text = stringResource(id = R.string.turns, turns),
                textAlign = TextAlign.Center,
                maxLines = 1,
                modifier = Modifier
                    .fillMaxWidth()
                    .padding(7.dp)
                    .border(2.dp, MaterialTheme.colorScheme.primary)
            )

            if (drawing != null) {
                DrawBox(
                    drawingZippedJson = drawing,
                    onClick = onGotoGame
                )
            }
        }
    }
}

@Preview
@Preview(uiMode = Configuration.UI_MODE_NIGHT_YES)
@Composable
fun PreviousGamesScreenPreview() {
    AppTheme {
        PreviousGamesScreen(
            onBackupGames = {},
            onDelete = {},
            onGoHome = {},
            onGotoGame = {},
            onImportGames = null,
            games = Json.decodeFromString(
                """[{"game":{"id":"16582a96-e0aa-4060-b879-33ce2d62eaa6","timeout":null,"turns":null},"entries":[{"id":"9a66c0e9-cf9a-4358-b6a3-42cf9c43f691","playerId":"3f82da6c-b576-4a7f-b1fb-d60232c7df24","sequence":0,"gameId":"16582a96-e0aa-4060-b879-33ce2d62eaa6","timePassed":0,"sentence":"A dog eating from a large bowl of food"},{"id":"aae44f43-35cc-40b9-a74f-0cf7fb0d5100","playerId":"3f82da6c-b576-4a7f-b1fb-d60232c7df24","sequence":1,"gameId":"16582a96-e0aa-4060-b879-33ce2d62eaa6","timePassed":0,"drawing":[31,-117,8,0,0,0,0,0,0,0,-35,-99,-37,-50,37,-57,109,-123,-33,101,-82,-123,66,-99,15,126,-115,0,-71,49,116,97,-64,3,75,-128,34,3,-106,-116,36,16,-4,-18,33,-9,-81,-111,-86,118,115,-83,102,35,-29,36,-56,-99,48,-30,-6,-2,-34,-35,117,96,-79,88,-84,63,-2,-14,-23,-121,-17,127,-4,-4,47,-97,-1,-14,111,-97,127,-4,-7,-89,79,127,-8,-29,47,-97,126,-6,-7,79,127,-5,-7,-45,31,126,-7,-12,31,-1,-6,-89,31,-2,-2,-7,-45,31,102,10,107,-43,84,-65,-7,-12,-97,-65,-2,75,-53,53,-84,-47,91,-1,-57,55,-97,62,-1,-8,103,-81,-15,63,-66,-7,-17,-46,71,15,37,-9,-68,-39,-74,20,74,44,-51,102,99,-13,11,-71,15,121,-116,-39,118,-37,33,79,-106,51,120,108,98,127,101,-57,-80,-26,-52,115,-77,93,45,-12,-98,23,96,99,-5,11,-69,-43,80,74,-34,30,-93,-89,20,114,-113,-32,-79,-79,-7,-123,92,-89,-4,-62,-108,-45,102,-101,-27,-63,-58,-102,-128,-51,4,46,122,-106,-97,93,-85,31,-2,-85,-3,-107,45,-19,103,-27,52,-25,110,44,-33,-85,-74,2,-32,68,-16,-116,-2,-19,55,-97,-2,-10,-7,-89,-65,-2,-16,-9,-97,-65,-1,-21,-113,-86,-6,-18,-13,-9,127,-7,78,-2,-104,52,-25,111,62,-3,-5,-9,127,-2,-7,-69,-41,127,-65,-98,-29,-66,-21,-43,-82,-65,122,-20,127,-87,-123,-43,74,50,30,11,-102,-38,63,-39,79,78,89,27,95,-118,-71,123,-31,84,113,-27,-105,21,82,-55,110,56,52,-65,-110,107,-111,-26,62,-5,-47,106,106,104,11,116,60,98,127,101,-53,80,-46,-57,56,109,101,4,-24,115,-128,-47,-120,42,12,-2,10,-77,-92,82,-9,-66,-70,-62,42,24,-113,5,87,122,-105,63,60,-22,-15,44,-100,78,4,62,-6,8,99,-44,-8,4,-1,69,97,-16,-85,118,-3,-39,79,107,-36,-63,-103,-64,-96,-53,-17,-102,-91,87,63,29,11,-82,-12,33,-83,117,-54,-8,-31,-90,19,-127,65,-105,126,61,-45,42,126,58,22,24,-12,-87,-58,-75,-7,-23,88,112,-91,-49,26,-46,-56,-45,-1,-34,-119,-32,74,95,67,123,95,58,30,69,-97,-82,-94,38,73,4,23,122,78,61,-116,94,-3,-67,-107,9,-82,-12,50,67,-103,-30,96,-72,-23,68,96,-48,-75,-7,-50,86,-4,116,44,48,-24,-38,-64,-42,49,21,-36,-48,-79,-32,25,-3,43,79,-58,83,-6,-31,42,-87,59,29,97,102,109,123,-62,-113,-8,73,124,-85,84,107,-34,-83,-21,8,41,-51,5,-8,84,97,-52,-54,50,91,-107,53,118,-29,-59,30,-97,9,-116,-103,57,-121,52,71,-39,58,87,19,-33,52,-11,-119,102,102,104,111,-80,103,-24,50,15,-18,-2,115,-21,-30,-27,-82,-127,-32,88,96,-50,-53,93,124,-38,-3,119,54,58,74,17,-127,49,-62,-90,16,-77,-4,52,55,-99,8,-116,17,54,-54,-93,-92,92,118,99,-2,81,-103,-62,-32,-113,16,71,27,-21,1,-97,40,-116,49,92,90,-63,-104,-51,-113,39,2,115,-122,-120,113,60,121,59,76,113,29,-85,98,12,35,-58,62,-35,124,-86,48,-8,57,-52,26,-117,-1,-7,-87,-62,-32,-53,96,-108,86,-9,-65,126,-86,48,-8,61,-60,18,-21,120,-64,39,10,-125,-81,-77,119,75,15,-16,88,96,120,0,73,125,-125,-14,-32,-21,50,-123,-63,31,-22,6,-82,-28,29,25,-104,-32,74,-105,-59,116,105,35,-113,115,8,28,-78,-56,6,116,34,48,-24,50,-64,-26,56,-6,110,-52,-42,120,76,96,-48,101,-31,83,-25,57,-122,20,89,87,-59,5,-126,9,84,97,-8,24,57,-12,21,-113,121,-7,-122,-49,20,6,-65,-88,15,-77,30,60,63,83,24,124,93,-4,-84,-47,30,-16,-119,-62,-57,-89,43,97,-86,-16,-8,120,-22,89,-12,-38,81,-57,34,2,-113,-9,-37,42,109,-101,68,96,122,-18,98,60,14,-81,-24,-26,-35,19,-123,-113,63,66,-87,11,56,82,76,112,-91,-41,107,36,82,-66,83,-21,29,4,80,-104,-64,-96,107,0,111,-52,115,124,-102,17,-123,92,-119,-67,-63,110,33,-90,113,-52,108,26,98,88,19,54,26,-90,48,-8,-38,-126,-57,97,-36,-59,29,69,31,21,-102,27,100,-3,26,-78,60,-36,35,-53,50,-100,-52,-126,-66,40,17,24,116,-19,26,125,30,113,-21,-95,-61,54,124,45,88,-32,-94,79,93,21,36,56,73,49,-123,-113,95,-62,76,-64,-79,39,-10,62,118,15,117,101,52,-120,17,-127,65,-41,65,-93,-57,121,26,-77,49,-110,41,12,-66,118,-21,126,-76,-34,41,67,94,-116,21,126,88,-94,112,-15,-59,-73,-82,-91,35,-1,-128,8,124,-12,26,-90,-116,74,15,-24,-65,10,124,-12,41,-58,-30,124,61,-64,127,81,120,-8,93,28,-21,-43,-27,-57,-70,-7,-65,43,124,124,89,-73,-49,-2,-32,-37,-2,-82,112,-15,83,20,-21,5,-101,38,22,92,-23,77,-89,-124,118,6,-23,106,104,105,-96,-121,39,2,-125,94,116,125,122,-124,127,-44,117,68,-19,6,-102,27,100,13,-69,-42,126,108,52,-56,-109,21,-8,81,-119,-64,-96,-9,119,-33,-1,38,42,70,4,-49,-24,95,59,42,-74,94,-37,97,-69,43,-102,112,80,-116,24,-37,49,-79,39,-12,49,67,-54,49,109,31,-96,46,-74,66,34,-10,87,-74,-72,96,-22,-125,-19,-74,37,-108,85,81,-76,-115,9,60,116,-7,-27,41,-90,9,102,54,38,112,-47,-27,37,-26,-44,30,-48,-65,8,92,-12,-41,88,-122,70,87,38,112,-47,-23,-38,-126,9,60,-12,-95,-127,-65,6,124,45,98,-17,98,-89,-112,100,29,11,-90,76,38,-16,-48,-59,-93,-108,79,-28,-1,-92,95,-20,93,-20,28,70,-82,-27,1,-4,-117,-64,-54,-31,104,125,-116,-43,118,-21,-8,90,63,-95,23,-61,20,-42,40,-48,101,109,-68,27,107,48,87,-122,12,56,10,32,-5,43,91,-121,-93,-38,-53,110,-101,67,108,56,3,-123,8,92,116,-2,98,-120,-64,67,-41,121,28,-81,68,-103,-64,69,-41,32,69,-84,-88,-51,16,-127,35,-85,-88,-22,74,51,69,-44,81,-103,-32,74,-41,-26,91,-10,40,101,45,45,-116,54,10,120,-17,76,-32,-39,8,-86,-70,85,82,35,120,51,76,-32,-94,103,89,16,54,-76,-83,-55,4,87,-6,43,50,48,-14,97,92,101,-111,-112,-32,-124,-51,20,22,95,-90,-9,-66,-57,-7,100,37,-85,-13,-40,2,115,54,85,92,-8,75,-5,-38,-40,67,6,53,-55,-6,111,-75,8,-38,13,19,92,-23,77,-9,-92,-10,-51,-19,26,117,85,-65,64,-77,33,-10,22,123,-83,56,54,-33,-92,44,89,-99,-91,-114,-36,102,38,112,-47,71,-56,112,24,35,-10,46,-74,-4,-45,24,32,-98,70,-20,93,108,-22,63,50,-127,-117,-98,-119,-1,72,-20,-81,-20,-87,-103,112,-78,-72,-34,-115,-93,76,51,25,45,-30,-88,-30,-70,121,21,-109,-84,79,75,-34,26,87,-103,-30,-33,70,-44,-111,-104,-64,-96,-53,2,-20,120,18,30,-68,32,-10,6,123,-122,44,-82,66,105,15,-16,76,114,-3,11,73,6,-25,-34,-10,49,-29,-18,15,48,-123,-79,21,-97,67,43,-79,-25,122,-66,75,-103,-27,-111,59,-58,37,-58,95,-104,47,119,-22,52,38,115,55,19,24,91,-3,37,-120,31,27,-121,31,-49,20,38,-1,45,11,-56,-63,-121,10,-125,-33,53,121,44,-9,7,124,-94,48,-8,-29,-61,49,-39,-83,117,-120,-54,40,-104,76,21,70,-62,-126,38,-65,-55,-54,-48,-33,62,-103,-62,-32,-21,-97,62,-110,-107,-117,-72,48,-71,64,23,-124,42,-116,-108,8,-7,95,49,-18,-77,102,-103,108,-32,100,2,-125,46,46,69,-110,-50,-72,27,83,-73,-104,42,-52,100,69,89,104,-19,93,101,-16,-58,67,4,102,-86,98,107,125,-49,-2,40,-93,-123,-36,-42,36,-71,-118,80,97,-16,103,-104,45,-19,-47,-78,50,-24,-108,72,21,102,66,-118,-4,-31,61,11,-18,-114,-49,20,102,58,100,26,115,-113,-102,-107,-63,-37,62,83,24,-4,21,-28,127,-49,99,36,76,-30,-93,14,-104,-118,-54,20,70,-54,75,-109,-59,-23,-37,52,90,66,-53,-67,-62,-84,69,-94,48,83,106,-92,-89,-97,-42,-76,111,17,-127,-103,80,19,101,-80,122,27,103,99,-116,9,37,98,49,-123,-103,82,-77,-6,-111,-42,91,52,10,-42,50,74,-18,38,2,51,-95,99,-108,99,7,-90,-120,-41,84,-60,11,67,-37,77,76,97,-92,92,104,68,119,79,-47,-84,-102,-48,66,114,58,-95,-67,-117,-83,-81,49,-95,97,-127,9,60,-55,34,85,35,-12,13,110,11,19,-127,-99,44,82,-30,110,27,121,-90,11,17,24,116,29,-1,98,-33,-93,73,-119,-89,19,48,-123,-63,-65,-8,22,53,105,-105,68,-89,2,-104,-64,76,117,89,113,-19,41,103,53,-21,-106,99,65,-127,98,-86,-16,36,67,84,77,72,-24,9,29,-7,-95,10,23,-65,-26,80,-37,-120,-88,97,50,-123,-105,79,-46,-104,-120,-64,71,-89,-77,21,85,-8,-8,124,27,-105,41,60,-23,28,-11,117,106,4,-19,3,48,-127,39,-99,-93,-74,22,102,95,40,28,66,21,62,-2,96,91,1,84,-31,-29,83,87,-115,42,92,-4,-98,67,111,79,30,-1,-117,-64,71,-17,124,27,-102,40,92,-4,33,-1,-76,22,-38,-26,-95,10,31,-65,-117,43,-10,-124,-2,97,-17,98,-117,-45,82,106,-119,48,-65,-128,40,124,124,-51,60,46,112,-44,100,10,71,42,83,93,49,-12,-103,50,124,57,88,-32,-94,-41,48,70,65,-15,105,38,112,-47,27,91,94,49,-127,39,-55,43,106,16,11,6,73,-103,-64,69,47,-102,-6,-124,-13,-115,-80,-64,67,23,-97,-88,-29,-103,28,-37,123,18,-56,82,-42,105,7,101,29,98,123,23,-69,73,-45,77,48,-97,3,-38,-69,-40,55,67,60,22,-72,-24,-30,-86,16,-25,-125,8,60,-12,28,89,-90,39,-74,119,-79,101,-35,50,75,-10,-65,-105,-33,4,-82,36,-43,-52,95,59,83,24,124,117,-1,-41,-18,-95,-73,-52,95,60,83,-72,-46,84,11,-9,-6,-80,-64,71,127,-83,-106,-54,-125,36,-37,-33,20,-58,-73,-43,-120,-58,-36,-29,-39,-83,-36,36,-17,17,-123,43,-51,-74,-36,52,123,-94,-16,-67,31,-18,55,49,-59,67,-2,87,-50,55,-54,-38,-65,-89,12,-75,-37,18,-66,-30,-59,30,-77,6,43,-31,103,-4,-9,85,121,-47,-52,-124,5,-113,77,96,123,23,-5,-26,-47,-79,-64,-96,107,-12,-96,-18,103,32,74,-25,-103,-23,76,97,44,-124,99,-120,-85,-92,-28,-25,51,-123,25,38,40,115,29,33,-10,-50,-3,22,-90,48,23,-14,109,-106,-8,-124,79,20,70,-89,81,-65,35,-113,7,124,-90,48,-121,-12,25,-113,56,-35,29,30,11,60,93,-66,-12,-86,-23,-67,15,-122,-36,-33,21,-98,33,81,-37,114,-85,5,123,-44,68,-31,-30,-53,-62,54,39,28,-36,99,10,31,63,-122,58,59,118,-62,-120,-62,-59,-81,93,-98,38,-11,7,-17,-25,55,-123,-113,79,-77,-6,-88,-62,-57,-41,98,49,-29,17,-1,-117,-62,104,-97,53,-52,28,-25,58,-97,38,-95,12,54,98,111,-80,-29,101,-33,95,-34,100,-83,21,-74,77,-90,48,70,30,89,-29,-41,-75,-41,-70,41,-107,-98,-12,35,2,-125,-82,-109,68,92,-35,79,-57,-126,103,-12,-81,-19,9,24,45,34,63,113,-78,126,-73,118,-73,56,-58,-65,28,7,-55,75,39,28,124,86,10,11,-116,-116,-18,-92,-101,118,-5,73,-29,60,53,41,126,54,-104,48,78,20,6,95,119,65,-10,61,-81,-84,126,73,44,-88,81,96,-5,43,91,38,-15,60,101,-99,-67,25,-21,-61,-115,1,15,-113,48,-123,-63,95,-95,-117,-25,-69,-7,61,122,-104,-85,-57,-40,-48,-85,103,-118,43,127,-108,48,123,-37,-117,37,-23,-22,42,-30,-29,23,68,96,-46,87,95,123,34,-56,-21,120,103,78,48,40,-53,20,6,-65,125,52,-39,-51,90,22,106,-93,-62,-120,56,17,24,116,45,52,18,-9,-118,98,58,4,-76,-98,-32,-79,41,-90,48,-8,-22,-115,-81,-13,97,74,-88,-30,18,66,62,81,-8,-8,-111,-91,118,80,-123,-117,-1,42,34,3,-13,98,-87,-62,-57,31,31,-121,63,30,-16,-65,40,12,-66,-26,91,-58,99,20,73,41,-92,-38,38,10,-102,48,-123,-63,-81,-30,-45,-98,61,49,86,77,-112,78,-112,79,20,-98,-34,-91,91,-52,-78,10,72,104,100,99,10,-125,31,-125,12,29,-5,72,-107,-106,120,-99,17,-58,-15,-120,-64,24,-39,-28,-121,29,5,-61,62,-110,19,22,-38,62,103,2,115,-52,95,101,-81,99,-105,100,-36,-99,-79,100,-44,116,-120,-64,-96,-57,48,-10,86,-106,-76,-42,76,-123,41,71,-60,-34,-16,-64,90,24,50,89,110,-58,-30,97,-107,4,-29,-31,-40,-34,-12,28,-91,-113,-20,-37,-79,-102,76,119,115,90,30,42,12,126,126,-17,31,73,-68,-118,20,59,74,72,-95,10,-61,3,-45,-60,-6,-72,-41,-31,73,57,125,56,50,-56,-63,35,10,-113,-9,-85,-123,-123,-118,44,-124,-32,-111,115,-94,112,-15,-45,-51,-5,39,10,-5,-3,-73,22,-33,-98,-26,-75,-50,101,31,0,74,124,95,-72,-68,-14,-54,73,-67,8,-88,48,91,-24,-37,8,-98,62,106,97,-110,-102,90,76,98,-58,14,74,-23,123,-92,-25,-2,47,48,-119,-71,-75,82,106,58,-102,28,-33,-89,39,2,115,-20,41,90,56,-57,-113,103,10,-45,-93,29,61,-18,-25,43,-17,-8,76,97,-50,-70,-93,46,63,29,-37,95,-39,122,66,-93,-25,7,-113,78,4,87,-6,42,-95,-43,99,118,-69,-95,19,-127,65,-41,-77,90,-67,-65,-115,-76,125,-32,-59,43,83,92,-8,69,-113,99,-44,81,-34,38,10,105,-43,96,98,97,-126,43,61,-23,12,119,-50,66,53,-116,-39,-48,41,119,38,48,-24,-102,22,-39,-10,-35,-57,-113,44,100,-24,9,82,-123,-63,95,-102,-122,-73,-65,-56,66,-109,-66,-104,-32,74,-49,-102,1,121,44,40,-17,-16,76,97,-16,101,-3,85,-10,-100,81,-83,40,-41,-5,66,-18,8,19,92,-23,26,61,58,-54,-128,37,-66,-55,-64,4,87,-70,-4,-81,56,-9,-80,114,-6,-75,6,49,106,55,68,96,-48,-89,44,-114,-54,-15,67,87,72,11,-83,79,-120,-3,-107,-83,1,-46,53,-10,19,-42,90,4,87,-1,9,-67,24,-90,48,-8,122,70,38,-42,126,90,-77,23,-49,20,87,126,-105,73,39,-114,-78,-65,-55,-90,101,38,-26,64,125,-118,41,12,-66,-84,53,-22,56,28,11,94,-103,-119,42,12,-66,-90,-15,-73,-42,31,-16,-119,-62,-32,-113,87,30,97,121,-64,39,10,-125,-1,58,3,-11,-128,14,-19,61,108,-23,-15,110,-82,-38,-6,-104,108,12,-61,-10,-113,-40,95,59,-80,-70,94,-89,99,-74,54,-96,11,40,-76,19,68,-116,-63,-108,-4,-124,-82,-89,102,-50,24,108,-26,-89,-106,-87,-62,-12,-122,-50,-108,-99,91,60,22,24,-2,-124,-90,-24,-108,-3,24,114,110,-103,-50,-55,76,97,-6,43,107,-28,35,-48,-88,-53,-24,-46,80,25,125,-86,48,-8,50,-78,-41,-29,24,89,-26,39,-46,-87,-62,-32,107,-101,78,-3,120,-101,52,-83,-123,42,124,124,89,-84,-53,-70,10,-51,15,76,97,-8,68,41,-44,-76,-114,48,-78,-98,-21,27,112,122,32,2,-45,95,60,-89,66,-35,-111,-88,-78,76,68,83,51,17,-104,-33,54,-49,-66,-114,126,72,-101,62,85,24,-4,22,-6,81,-75,-4,22,-113,5,6,93,-100,-126,-36,-54,112,15,59,84,97,-82,52,100,-112,58,-34,-91,-8,108,81,-6,33,28,-42,-120,-62,-32,-21,60,91,-9,-104,-47,43,-122,79,6,30,-90,-8,-97,30,-107,25,-3,43,79,70,-81,-109,123,-25,81,-45,-124,-21,74,83,107,-69,93,124,69,-2,-41,-2,-23,-81,99,115,99,-81,-19,-84,65,81,52,25,48,107,-5,-89,63,-29,107,-7,2,45,-39,-79,-121,127,-109,86,52,64,69,-69,-87,-62,-32,47,25,27,-9,35,-41,105,-60,-112,18,92,-37,51,-63,-123,94,-93,70,52,-113,123,14,-12,-22,-125,94,-54,4,83,1,85,24,-4,17,102,29,123,114,-111,94,79,48,-30,68,-79,3,-86,48,-8,-38,-54,-42,-38,23,-21,-21,-27,43,-96,12,121,-86,-72,-14,83,-106,111,-75,-25,94,-23,-82,71,26,-16,-16,0,19,24,-12,-41,121,-27,125,-20,-110,-73,91,26,44,35,-62,4,6,93,-85,13,28,27,-23,50,-115,55,89,-18,-62,103,-57,2,-125,-34,-61,-88,-79,-17,-93,93,106,-95,-114,-123,-86,125,82,-123,-63,95,-17,-67,-16,117,-96,109,44,52,-43,80,-59,-107,-81,97,-40,-74,-50,-35,-24,-92,117,-23,-32,-53,103,10,-125,63,67,-108,53,-35,-101,91,48,38,-38,-92,98,-126,43,93,-2,-80,-52,122,91,51,83,127,111,-108,-118,-118,19,50,-127,65,-17,-95,-55,119,-33,91,66,-45,-89,91,-88,-80,16,85,92,-7,-30,59,38,105,-75,-5,79,29,122,-31,22,-38,106,96,2,-125,-34,-61,-100,-121,105,11,-77,73,-61,64,112,104,111,-80,47,55,-73,100,-103,-5,-30,44,104,-76,-89,10,31,63,-53,123,-124,-29,13,17,92,-23,77,-4,-20,60,-42,-2,-107,38,117,-81,-88,-62,-32,55,-23,110,113,-20,61,80,-85,119,-64,-53,100,-104,-32,74,-105,-91,126,110,-89,-93,-67,-76,-98,26,108,-13,68,112,-91,47,-23,32,-85,31,-117,16,-11,93,35,-54,55,97,-126,11,-3,117,-4,-94,-18,-25,-120,-43,-81,111,117,-95,92,31,38,-72,-46,-85,46,65,-114,-77,-19,50,-108,-9,-124,-114,-18,18,-5,43,-69,47,45,51,119,84,-115,-32,27,-113,84,113,-27,107,25,-69,-30,-121,67,115,-125,-84,71,-2,-9,34,-103,-73,108,44,48,-97,-5,45,100,80,50,13,2,80,-59,-107,-81,-59,15,-26,58,-78,-20,-76,80,86,78,-24,14,67,34,-72,-46,87,9,-79,-100,-27,16,-14,10,101,102,84,-21,-118,42,12,-66,-12,-30,46,99,-36,102,93,94,-41,82,-95,-125,63,84,97,-16,-11,83,-19,-65,-75,-16,119,-113,-19,13,-10,-59,-113,-72,-91,19,-59,-123,-33,-93,-106,78,43,71,21,34,-50,-89,-118,43,63,-59,80,100,126,-39,-65,-108,44,-119,75,-22,104,-59,67,21,87,-66,54,-38,126,-44,-94,40,31,-27,41,-48,-91,-96,68,112,-91,-53,31,-114,105,-113,-12,105,-90,108,-127,53,-28,-119,-3,-107,93,-117,-18,85,-19,77,-84,-59,48,-31,121,34,98,127,101,-53,-1,-119,-69,33,-99,87,-95,-11,61,-73,-79,2,-72,-48,-6,-98,-53,-110,-99,-111,-15,61,85,-90,-126,50,-31,-73,67,-42,30,46,9,94,64,-21,-1,-67,-73,-16,-49,121,-38,127,-38,86,-57,43,-72,-34,-50,1,-127,-28,-107,16,107,-112,87,-14,-120,127,-51,-51,-44,-31,111,66,-113,-98,9,-116,-67,-114,-2,126,40,-79,-36,28,34,101,10,-57,94,74,-71,73,-100,35,2,-125,-66,-34,-50,-53,-34,-46,-79,-64,-120,104,-90,-9,100,-4,-14,58,-54,-128,-68,4,38,48,-94,-43,81,-36,-48,-71,-114,-9,-88,21,93,39,12,-99,49,-123,-71,15,-44,-50,35,-117,-30,85,-41,56,80,-28,12,-37,27,-69,16,122,-105,72,47,-121,-109,59,100,-71,-111,35,10,-28,51,-123,-63,-105,69,-87,-8,92,123,43,72,50,-45,-83,-118,-50,30,80,-123,-111,-7,-95,23,-28,-55,-70,122,-73,-26,123,88,76,97,-16,-43,-9,60,60,22,121,-69,-93,-63,-72,28,19,24,-12,-105,59,23,15,-21,-92,-95,30,-40,114,-104,-62,-52,-117,121,15,37,-83,-101,-68,27,-94,48,51,99,-28,127,-107,-20,-25,51,-123,-63,-41,75,90,-13,113,82,100,-15,-96,55,83,24,124,-3,-45,-5,21,112,-70,-9,17,71,67,-53,89,38,48,114,111,114,-120,-79,-108,99,-35,46,14,82,-55,40,87,-105,42,12,126,13,57,29,103,58,95,-29,22,-55,-113,97,-118,-121,-4,-81,-67,27,82,-45,-5,-90,19,46,59,-62,-116,97,-50,-108,-97,46,115,-84,124,-48,99,117,94,67,89,5,5,104,-104,-64,28,81,100,-59,112,20,-104,-108,-123,-123,124,103,116,115,0,85,24,35,-82,-18,84,-26,61,-57,-25,53,51,-74,-122,10,-101,80,-123,-63,-81,-95,-74,99,-95,34,75,-105,22,-47,-119,11,98,111,-17,-24,-89,-77,-76,112,21,95,-79,-61,50,68,84,-31,-55,-88,-48,-109,-102,45,101,20,-39,-93,10,51,35,36,-55,18,-15,109,-91,37,-125,27,-53,8,-127,10,-45,-45,88,-93,-20,101,-11,75,-85,97,-108,1,-109,6,-104,-62,-28,-57,89,-113,-106,-36,-90,94,-90,56,-119,39,3,21,-98,125,119,77,121,-117,112,43,-121,9,124,-12,38,94,-43,-128,85,23,-103,-62,-57,-105,127,-86,9,110,-22,99,-127,-117,62,114,-104,43,85,116,-14,-126,41,124,-4,-58,-118,89,16,-127,-113,-2,17,28,121,64,-1,85,-32,-56,72,-72,-87,34,-53,4,-66,103,-65,-63,19,-123,-113,127,-109,54,79,20,102,46,78,-116,71,22,-68,-74,3,28,0,96,2,59,75,-20,-88,4,116,7,-121,-10,102,86,-69,44,-78,-10,77,-85,-46,111,-46,-50,-119,-62,-104,107,-59,-97,-102,105,-49,58,-46,20,85,61,-96,-118,-58,123,-90,48,60,5,25,-67,83,95,-29,-76,38,47,-121,8,76,47,-89,-90,-76,-17,42,-35,20,-7,-96,10,-125,-81,55,-6,-54,112,125,90,-89,-34,-47,94,17,85,24,-7,-37,69,51,-57,-5,-15,52,57,-116,-102,-31,-127,11,-90,48,-13,-49,-91,17,31,37,77,110,-46,-61,-79,-64,-109,125,-2,-6,39,124,-74,-105,42,-52,-20,-13,124,-58,100,111,-78,-119,-95,-67,-15,-20,-70,74,60,74,-43,-36,-64,-119,-64,69,-89,-115,30,-37,27,111,69,55,-16,-50,-14,-21,28,78,4,118,70,-5,91,-11,-20,27,58,22,-36,-25,112,-33,-79,-111,-71,-111,-4,-107,67,-109,73,97,-70,-39,68,96,-46,-33,99,10,-81,126,-79,80,-103,112,-86,48,-8,-38,-85,-113,-70,82,50,-116,100,84,-66,1,-101,123,-110,-30,52,-123,47,105,1,124,119,82,-36,-17,10,-125,63,-76,-92,-58,17,-51,89,-14,102,7,92,-20,16,-127,65,87,39,101,-60,-29,-58,-96,26,100,-72,64,14,37,85,-72,-8,73,-34,109,38,111,-121,40,-52,-73,31,101,101,119,92,-89,-76,-28,105,-26,64,-29,59,83,24,124,93,-76,-49,-13,-78,-87,-105,-105,5,31,31,11,-116,-34,42,127,-72,31,-91,-107,-117,-42,-81,-59,-95,37,34,-16,-116,52,-81,-78,-36,-16,-124,45,19,-36,-113,52,55,55,-55,17,123,-125,-35,-33,114,-13,110,-18,-17,99,2,115,124,127,75,91,-44,-12,22,122,-22,-121,40,92,124,-23,-109,121,-96,-5,26,-104,-64,69,-41,-78,-60,99,-94,-79,-116,8,124,-12,-87,-25,103,-95,75,-128,5,-90,-65,-15,118,105,-115,-34,-121,90,103,-124,-111,126,-90,112,120,5,45,-22,122,46,-93,-101,25,-103,-64,-12,-11,-50,-61,15,77,111,110,-45,-92,75,-20,-21,33,-127,65,-81,50,-61,28,21,46,-45,12,57,-62,-88,19,-74,55,-40,50,-2,31,25,-117,77,-17,75,32,103,36,-95,-67,-63,78,-17,-47,-87,-106,-27,-47,-48,70,53,-79,55,-40,90,-54,100,15,-44,55,61,90,-67,-32,-36,-124,-19,-51,-107,-57,58,-50,85,55,-51,64,-83,112,93,-128,-19,77,118,-102,115,-97,-61,120,49,90,38,48,-24,69,126,-45,72,-57,-93,-56,112,81,-15,-127,90,44,48,-24,-81,42,41,-5,-42,93,-45,127,-22,21,69,-80,-88,-62,-36,-17,16,23,-27,-72,117,-66,-14,51,-105,76,97,-18,-41,-84,-79,7,-110,-101,-68,-36,8,-85,9,17,123,-125,-83,57,32,-5,-42,72,-45,36,-106,-118,82,121,-119,-67,-63,46,31,-117,-99,-35,-8,117,7,26,-34,-57,34,10,31,-65,-123,-42,-16,-114,2,22,24,-12,-12,-18,-75,53,-67,35,-85,69,20,124,-93,10,31,-97,-59,61,-103,-64,-77,71,-39,-8,-38,-122,8,-52,29,-42,-47,-114,-5,127,-37,-51,82,-104,41,12,-66,44,63,91,-38,19,-117,111,-7,68,97,70,-80,-70,-26,14,110,-42,26,8,-24,48,-32,79,4,70,-4,109,5,-23,-55,71,33,-21,-101,16,10,83,-104,-69,-1,-78,40,60,-54,-96,-21,17,-95,2,-41,126,68,96,-48,-75,44,-34,-111,125,115,-9,-18,-103,-62,-32,-89,80,-6,-66,-39,123,-117,-57,2,51,-82,-102,71,126,64,39,2,-125,-2,-111,3,-4,-96,93,50,-123,103,39,-19,-106,79,20,6,95,11,36,29,-51,-20,14,-113,5,-10,62,-102,-76,-87,7,116,44,48,-29,-3,-30,104,-6,-101,37,19,120,118,19,-76,-125,-77,60,47,-90,112,-20,-75,-76,-41,-107,-123,104,1,69,-20,93,-20,30,-78,-52,59,-18,125,-100,-33,5,46,58,45,125,-117,-19,93,-20,-101,111,-118,5,-66,111,122,-37,100,-96,-62,104,-17,50,-25,-116,116,120,113,55,-5,44,76,97,-114,54,-67,62,-62,99,-127,-103,-93,-74,86,-102,-2,-73,67,21,-58,28,-85,1,-57,99,-113,-4,-114,-49,20,102,-106,84,61,-74,-8,-17,-16,68,96,-48,-75,37,-76,99,113,116,-121,39,10,51,-117,73,90,-38,-98,-17,-41,-58,77,-106,17,81,-104,89,70,75,-102,90,127,-64,39,10,123,29,91,-114,-14,-112,119,124,-90,112,-83,-63,-17,-8,68,-31,-119,124,-36,-31,-79,-64,19,-75,-71,-93,99,-127,17,115,-118,-17,-11,-62,-17,-16,76,-31,-30,-33,-76,125,-90,-16,-60,-52,110,-7,68,-31,-29,-13,61,35,44,-16,-60,19,117,-34,-113,107,-94,74,-10,84,-31,-120,20,-85,-81,-34,59,44,92,-122,-19,-19,8,119,61,29,-82,87,-26,43,45,14,5,4,-26,-50,66,25,-71,31,-66,110,121,-99,-37,-62,59,11,72,96,-18,119,-11,49,-9,-12,-118,-41,73,-59,-127,106,17,48,-127,65,111,-95,-42,114,-122,-85,102,-120,5,38,84,19,-63,-107,62,-93,22,-30,60,-81,30,-118,90,90,5,125,84,34,48,-24,-81,-53,-81,-49,107,-79,-102,102,74,-94,-81,74,4,6,125,6,89,1,-28,35,44,-85,-55,40,-24,-75,99,-5,43,91,-45,112,103,-39,-101,-105,120,72,-71,39,-28,-71,50,-127,65,-97,33,-115,-72,55,0,-115,-76,-113,54,-47,49,67,-86,-72,-16,95,-59,46,98,58,-82,23,94,57,-92,12,-49,75,19,-127,65,-81,33,-10,-3,-89,-42,-103,-62,-56,25,-35,-45,-52,4,6,93,47,117,78,123,77,-68,58,94,117,-80,33,29,11,-82,116,-15,20,103,-36,-101,-128,-34,-63,-103,-22,-60,-91,43,-80,-64,-96,-53,-93,-84,116,-20,-96,52,-7,112,-46,20,-48,89,111,-90,-72,-14,115,-46,93,-121,-78,27,103,105,-45,-88,-78,4,52,55,-56,-105,0,100,-83,67,-6,34,-68,-13,-110,42,12,126,-1,88,-128,-18,-42,26,63,93,104,-77,-117,42,12,-66,-68,-57,35,17,-86,86,-70,-117,-55,4,38,93,-98,101,15,46,-21,94,112,-123,17,81,38,-16,-47,101,98,110,-72,-40,12,83,-8,-8,-68,124,2,83,92,-7,37,-123,-38,-114,77,-8,21,90,66,-23,-19,-40,-36,32,95,-50,-37,85,77,-114,47,5,-19,97,82,-123,-113,79,99,43,84,97,-16,-43,53,95,-3,-8,-79,-81,91,-114,80,-93,39,2,-125,-2,58,-24,-78,-89,66,125,-44,-110,-127,-34,42,85,60,-28,127,-5,117,-49,-125,-44,-41,-123,98,-57,28,-111,96,38,35,51,6,63,-4,9,-67,107,-3,-19,118,100,102,52,-83,-125,-120,70,66,108,127,101,-117,115,86,-26,-38,75,65,-24,-3,-25,125,32,127,-117,9,-82,116,13,44,-114,-38,-113,17,63,-78,-66,72,4,23,122,123,-91,121,28,-37,-78,122,63,54,94,-67,48,-63,-107,62,-12,-74,-92,99,64,-106,-119,-74,15,-108,91,68,-20,-81,-20,-87,59,-127,37,-99,-74,100,89,-54,4,6,-3,-29,122,-35,-35,-1,72,33,-74,-118,-26,101,38,-72,-46,-105,-98,-65,60,110,-100,127,-3,114,82,-43,1,11,46,-12,30,-11,-8,-30,97,59,-59,-65,71,-57,-12,-111,-11,-107,-85,-43,-56,-53,-111,60,53,105,32,-128,9,12,122,121,69,67,-9,7,-31,103,-43,-119,-64,120,39,51,72,-97,-34,99,6,119,15,-49,20,-58,23,93,-31,-88,98,-93,73,60,41,70,116,48,-125,-40,27,108,-103,-20,-38,-111,-63,19,-125,86,-33,-127,108,104,111,-74,-60,88,-41,97,-84,5,62,-32,-102,-120,9,-116,94,-92,7,-68,119,-45,-82,-41,61,-94,-59,28,54,55,-56,-70,26,56,50,108,98,-44,43,-67,10,44,73,-125,5,-58,-72,-91,-9,-83,28,-101,-128,-78,-36,89,5,118,34,38,48,-24,122,-103,86,-34,-121,57,-67,-123,92,119,29,33,-98,40,12,-66,118,-116,-68,47,91,-45,71,-90,13,28,118,-79,-64,-104,49,46,-101,34,-119,30,-124,97,2,-125,-98,-33,-77,73,-24,5,-32,-60,-34,-98,-21,-114,-85,-105,52,43,-21,102,-86,-77,-19,-81,-20,-41,-71,-127,51,-39,-125,-33,93,72,21,6,95,-81,124,59,54,43,-54,-21,46,114,4,71,-26,6,89,23,1,-3,72,14,-46,97,14,-50,-47,-40,-34,-59,-82,-78,54,70,-51,28,-102,123,-34,71,-19,-102,-4,-126,-94,80,76,-32,-95,55,118,103,52,-79,55,91,74,-20,-27,-80,-107,79,-45,80,-118,37,-79,55,90,-72,-18,74,-20,71,123,90,-21,36,39,-113,-40,-101,61,115,-28,116,-90,51,-79,56,55,19,24,-12,114,-39,-49,-70,-95,99,-127,103,84,-31,-69,-7,76,-32,-95,119,117,-50,80,20,-124,-40,-69,-40,61,100,-72,-39,78,-20,93,108,113,-3,82,-126,101,-35,-120,-64,-96,-53,34,-89,-98,59,5,-4,-125,66,123,115,14,42,-29,-63,94,53,19,24,-12,-7,-26,88,-22,-10,-107,76,41,19,-46,-79,-64,-104,-99,47,-89,-71,117,-2,-19,-16,-62,18,38,-16,-47,-59,-111,-59,-73,101,82,-123,-63,-65,-36,126,-48,-8,57,91,-86,-16,-15,111,42,37,18,-123,-7,126,82,-118,79,30,31,11,-52,118,89,-5,81,109,-29,14,-49,20,6,-65,-122,92,-9,115,85,-73,120,44,48,102,38,-67,80,41,-73,-30,-57,51,-123,-63,23,23,60,-42,-30,-1,-76,68,96,-48,-93,-122,-16,-45,19,60,81,92,-7,122,-3,-16,81,-53,92,3,10,-60,51,-59,-10,6,91,-61,-115,-57,-40,119,-61,-122,-10,6,59,-55,-62,39,62,96,67,-5,43,-5,-107,89,126,108,49,-67,74,127,86,-28,111,96,123,-125,61,66,42,-57,-94,100,82,111,29,-37,27,-20,75,122,-59,-28,13,-99,8,12,-70,-70,38,71,106,-56,29,29,11,-82,116,61,-111,113,54,-38,27,58,17,24,-12,113,25,77,-17,-16,68,97,-16,91,24,115,-74,7,120,44,48,-24,81,124,-99,121,-4,-44,21,-60,49,71,-59,-121,-79,-3,-107,-83,87,48,-25,-102,-113,14,-51,-57,23,-90,48,-8,37,-116,18,-113,-68,-26,69,59,42,17,92,-23,122,-43,-64,121,110,94,-85,-24,-30,-4,127,38,48,-24,-13,125,-31,-74,104,95,-59,-10,6,91,63,-9,-111,61,112,-61,-122,-10,-58,59,23,55,45,-113,-26,-121,19,-127,49,14,104,-99,-57,-6,-2,6,51,116,-12,-80,-67,49,-86,23,-79,61,38,-10,-59,-93,48,68,96,-52,-91,-14,51,-45,113,14,108,-35,56,2,88,96,-82,-128,-59,49,-39,-66,126,-113,-76,-95,99,123,51,-58,35,83,-6,112,-77,-79,-67,35,126,116,-49,6,-10,-114,-75,-40,29,27,-38,-101,43,-79,-76,-10,73,-9,-114,13,-19,-51,-75,82,41,123,10,75,-113,60,-90,6,-19,-51,-75,64,-87,41,-71,-39,-40,-34,96,15,-103,20,-41,3,54,-76,55,-93,-57,61,-51,-22,103,67,123,115,-41,-21,116,68,110,-40,-40,-34,-116,-87,-113,-29,26,-31,59,54,-76,55,-40,50,28,28,-7,107,119,108,104,111,-77,-113,122,60,14,-74,109,127,-69,-125,113,71,6,-42,-58,-66,-53,-5,-19,57,55,100,108,111,-80,-43,-99,92,126,52,50,55,-56,111,-91,-21,-17,-56,-56,-36,-40,-97,-45,-15,-84,-69,-47,-60,-34,-36,15,125,31,-121,57,27,-38,-69,-40,108,108,37,-10,6,-5,-3,100,-19,29,27,-38,59,-40,58,97,-113,10,-117,-60,99,123,23,91,83,-44,-16,14,52,-76,-9,-80,117,-29,20,-98,-60,39,-10,46,118,13,89,-42,54,-80,-95,96,-127,-121,-82,-101,-122,13,-43,-111,37,-10,-114,86,-88,97,-85,90,22,74,81,100,2,-125,94,-11,-58,-28,99,-9,-111,-122,114,-103,-64,-47,-13,-17,-32,-48,-34,96,95,-113,96,-1,90,75,13,-63,-79,-64,71,-25,87,45,48,-123,-113,-81,-55,-10,13,-9,36,-94,112,-15,-59,-43,30,-113,30,-1,-117,-64,71,-89,121,120,84,-31,-29,39,13,71,62,-8,-74,95,4,102,-113,-22,-29,-36,118,-29,-73,84,82,-123,-89,-57,106,125,-116,-116,-74,70,-119,-67,57,-46,-24,117,-63,-69,45,111,-12,-48,-34,51,-118,105,-67,-21,-119,82,58,-120,-67,-63,126,47,-52,-35,74,15,115,-95,-99,81,98,111,100,23,-67,23,-113,-42,-6,32,125,-94,19,-79,-60,-34,96,119,-35,-67,-37,115,-88,52,-26,-42,26,-54,-48,99,2,-125,126,-115,-81,101,25,95,101,-80,-122,-49,78,20,6,95,-33,98,62,78,42,102,45,83,-120,-37,34,83,92,-7,89,-61,84,71,28,41,107,-79,-74,-124,10,91,48,-127,65,-41,115,-124,-15,8,-102,-56,-49,-81,-16,34,54,38,48,-24,51,-60,-44,-113,2,36,-119,6,66,-88,-30,-54,47,81,6,-25,89,31,-16,-103,-62,-32,107,9,-63,-13,-72,22,79,-42,-95,10,-125,95,52,65,61,63,-31,19,-123,-63,127,37,-64,63,-96,67,123,-125,-83,-29,-25,113,-6,-26,-10,-35,19,-123,-51,127,123,-111,-22,-65,-31,27,-98,-120,-64,-96,-65,85,-125,-29,-87,76,-40,-36,67,-66,121,-25,-48,-2,-54,-42,-29,24,-29,8,-26,-33,-64,-119,-64,-96,-25,115,-71,121,11,-121,-10,6,-5,82,115,-14,-106,78,20,6,127,105,-63,-89,-14,0,-113,5,87,-6,121,89,-45,29,26,89,-1,-1,-29,-26,48,115,71,-71,52,-48,-6,-1,-64,123,-8,-42,125,82,-29,-37,-1,2,23,-101,-89,-53,66,-28,0,0]},{"id":"52d5eba4-5ece-4154-9294-31bc3b867f23","playerId":"3f82da6c-b576-4a7f-b1fb-d60232c7df24","sequence":2,"gameId":"16582a96-e0aa-4060-b879-33ce2d62eaa6","timePassed":0,"sentence":"A dog looke deeply into a crystal ball"}]}]"""
            ),
        )
    }
}

@Preview
@Preview(uiMode = Configuration.UI_MODE_NIGHT_YES)
@Composable
fun NoPreviousGamesScreenPreview() {
    AppTheme {
        PreviousGamesScreen(
            onDelete = {},
            onGoHome = {},
            onGotoGame = {},
            onBackupGames = {},
            onImportGames = null,
            games = listOf()
        )
    }
}
