/* Copyright (C) 2025 Charles Lombardo <clombardo169@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 */

package dev.clombardo.dnsnet.ui.app

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.IntrinsicSize
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.asPaddingValues
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.layout.systemBarsPadding
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.LazyListState
import androidx.compose.foundation.lazy.rememberLazyListState
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.BugReport
import androidx.compose.material.icons.filled.Download
import androidx.compose.material.icons.filled.History
import androidx.compose.material.icons.filled.Info
import androidx.compose.material.icons.filled.Upload
import androidx.compose.material3.CircularProgressIndicator
import androidx.compose.material3.MaterialTheme
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.focus.FocusDirection
import androidx.compose.ui.focus.FocusRequester
import androidx.compose.ui.focus.focusProperties
import androidx.compose.ui.focus.focusRequester
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import dev.clombardo.dnsnet.ui.common.FabState
import dev.clombardo.dnsnet.ui.common.IconSettingButton
import dev.clombardo.dnsnet.ui.common.ListSettingsContainer
import dev.clombardo.dnsnet.ui.common.SplitSwitchListItem
import dev.clombardo.dnsnet.ui.common.SwitchListItem
import dev.clombardo.dnsnet.ui.common.TriStateFab
import dev.clombardo.dnsnet.ui.common.isSmallScreen
import dev.clombardo.dnsnet.ui.common.navigation.NavigationBar
import dev.clombardo.dnsnet.ui.common.plus
import dev.clombardo.dnsnet.ui.common.rememberFocusRequester
import dev.clombardo.dnsnet.ui.common.theme.Animation
import dev.clombardo.dnsnet.ui.common.theme.DnsNetTheme
import dev.clombardo.dnsnet.ui.common.theme.FabPadding
import dev.clombardo.dnsnet.ui.common.theme.ListPadding
import kotlinx.coroutines.delay

object Start {
    const val TEST_TAG_START_BUTTON = "start_button"
}

@Composable
fun StartScreen(
    modifier: Modifier = Modifier,
    contentPadding: PaddingValues = PaddingValues(),
    listState: LazyListState = rememberLazyListState(),
    resumeOnStartup: Boolean,
    onResumeOnStartupClick: () -> Unit,
    blockLog: Boolean,
    onToggleBlockLog: () -> Unit,
    onOpenBlockLog: () -> Unit,
    onImport: () -> Unit,
    onExport: () -> Unit,
    isWritingLogcat: Boolean,
    onShareLogcat: () -> Unit,
    onResetSettings: () -> Unit,
    onOpenAbout: () -> Unit,
    state: FabState,
    onChangeVpnStatusClick: () -> Unit,
    startButtonFocusRequester: FocusRequester,
) {
    Box(
        modifier = modifier
            .fillMaxSize()
            .focusProperties {
                onEnter = {
                    startButtonFocusRequester.requestFocus()
                }
            }
    ) {
        val columnFocusRequester = rememberFocusRequester()
        LazyColumn(
            modifier = Modifier.focusRequester(columnFocusRequester),
            state = listState,
            contentPadding = contentPadding + PaddingValues(ListPadding) +
                    PaddingValues(bottom = TriStateFab.size + FabPadding),
            verticalArrangement = Arrangement.spacedBy(8.dp),
        ) {
            item {
                ListSettingsContainer {
                    item {
                        SwitchListItem(
                            title = stringResource(id = R.string.switch_onboot),
                            details = stringResource(id = R.string.switch_onboot_description),
                            checked = resumeOnStartup,
                            onCheckedChange = { onResumeOnStartupClick() },
                        )
                    }

                    item {
                        SplitSwitchListItem(
                            title = stringResource(id = R.string.block_log),
                            details = stringResource(id = R.string.block_log_description),
                            maxDetailLines = Int.MAX_VALUE,
                            checked = blockLog,
                            bodyEnabled = blockLog,
                            onCheckedChange = { onToggleBlockLog() },
                            onBodyClick = onOpenBlockLog,
                        )
                    }
                }
                Spacer(Modifier.padding(vertical = 4.dp))
            }

            item {
                ListSettingsContainer {
                    item {
                        IconSettingButton(
                            title = stringResource(R.string.action_import),
                            details = stringResource(R.string.import_description),
                            icon = Icons.Default.Download,
                            onClick = onImport,
                        )
                    }

                    item {
                        IconSettingButton(
                            title = stringResource(R.string.action_export),
                            details = stringResource(R.string.export_description),
                            icon = Icons.Default.Upload,
                            onClick = onExport,
                        )
                    }

                    item {
                        IconSettingButton(
                            enabled = !isWritingLogcat,
                            title = stringResource(R.string.action_logcat),
                            details = stringResource(R.string.logcat_description),
                            icon = Icons.Default.BugReport,
                            onClick = onShareLogcat,
                            endContent = if (isWritingLogcat) {
                                {
                                    var entered by remember { mutableStateOf(false) }
                                    LaunchedEffect(Unit) {
                                        delay(100)
                                        entered = true
                                    }
                                    AnimatedVisibility(
                                        modifier = Modifier.height(IntrinsicSize.Max),
                                        visible = isWritingLogcat && entered,
                                        enter = Animation.ShowSpinnerHorizontal,
                                        exit = Animation.HideSpinnerHorizontal,
                                    ) {
                                        Row(
                                            verticalAlignment = Alignment.CenterVertically,
                                            horizontalArrangement = Arrangement.Start,
                                        ) {
                                            CircularProgressIndicator(Modifier.size(24.dp))
                                            Spacer(Modifier.padding(horizontal = 8.dp))
                                        }
                                    }
                                }
                            } else {
                                null
                            }
                        )
                    }

                    item {
                        IconSettingButton(
                            title = stringResource(R.string.load_defaults),
                            details = stringResource(R.string.load_defaults_description),
                            icon = Icons.Default.History,
                            onClick = onResetSettings,
                        )
                    }

                    item {
                        IconSettingButton(
                            title = stringResource(R.string.action_about),
                            details = stringResource(R.string.about_description),
                            icon = Icons.Default.Info,
                            onClick = onOpenAbout,
                        )
                    }
                }
            }
        }

        val isSmallScreen = isSmallScreen()
        Box(
            modifier = Modifier.fillMaxSize(),
            contentAlignment = if (isSmallScreen) {
                Alignment.BottomCenter
            } else {
                Alignment.BottomEnd
            },
        ) {
            TriStateFab(
                modifier = Modifier
                    .testTag(Start.TEST_TAG_START_BUTTON)
                    .then(
                        if (isSmallScreen) {
                            Modifier
                                .padding(bottom = NavigationBar.height)
                                .systemBarsPadding()
                        } else {
                            Modifier.padding(TriStateFab.safeInsets.asPaddingValues())
                        }
                    )
                    .padding(FabPadding)
                    .focusRequester(startButtonFocusRequester)
                    .focusProperties {
                        up = columnFocusRequester
                    },
                state = state,
                onClick = onChangeVpnStatusClick,
            )
        }
    }
}

@Preview
@Composable
private fun StartScreenPreview() {
    DnsNetTheme {
        StartScreen(
            modifier = Modifier.background(MaterialTheme.colorScheme.surface),
            resumeOnStartup = false,
            onResumeOnStartupClick = {},
            state = FabState.Inactive,
            onChangeVpnStatusClick = {},
            blockLog = true,
            onToggleBlockLog = {},
            onOpenBlockLog = {},
            onImport = {},
            onExport = {},
            isWritingLogcat = false,
            onShareLogcat = {},
            onResetSettings = {},
            onOpenAbout = {},
            startButtonFocusRequester = rememberFocusRequester(),
        )
    }
}
