package dev.bg.bikebridge.util.ktx

import android.Manifest
import android.app.NotificationChannel
import android.app.NotificationManager
import android.app.PendingIntent
import android.bluetooth.BluetoothManager
import android.content.Context
import android.content.Context.NOTIFICATION_SERVICE
import android.content.Intent
import android.content.pm.PackageManager
import android.location.LocationManager
import android.os.Build
import androidx.core.app.ActivityCompat
import androidx.core.app.NotificationCompat
import androidx.core.app.NotificationManagerCompat
import androidx.core.location.LocationManagerCompat
import dev.bg.bikebridge.ExceptionActivity
import dev.bg.bikebridge.MainActivity
import dev.bg.bikebridge.R

inline fun <reified T> Context.intent(): Intent =
    Intent(this, T::class.java)

fun Context.isBluetoothEnabled(): Boolean {
    return (
        getSystemService(Context.BLUETOOTH_SERVICE) as BluetoothManager?
    )?.adapter?.isEnabled ?: false
}

fun Context.isLocationEnabled(): Boolean {
    return LocationManagerCompat.isLocationEnabled(
        this.getSystemService(Context.LOCATION_SERVICE) as LocationManager
    )
}

fun Context.getPendingIntent(): PendingIntent = PendingIntent.getActivity(
    this,
    0,
    Intent(this, MainActivity::class.java).apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
    },
    PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
)

fun Context.getExceptionPendingIntent(
    exception: Throwable
): PendingIntent = PendingIntent.getActivity(
    this,
    1,
    Intent(this, ExceptionActivity::class.java).apply {
        flags = Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_CLEAR_TASK
        putExtra("stacktrace", exception.stackTraceToString())
    },
    PendingIntent.FLAG_UPDATE_CURRENT or PendingIntent.FLAG_IMMUTABLE
)

fun Context.sendNotification(
    channel: Pair<String, String>,
    title: String,
    description: String,
    pendingIntent: PendingIntent = getPendingIntent()
) {
    val (channelId, channelTitle) = channel
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
        (getSystemService(NOTIFICATION_SERVICE) as NotificationManager).createNotificationChannel(
            NotificationChannel(
                channelId,
                channelTitle,
                NotificationManager.IMPORTANCE_DEFAULT
            )
        )
    }
    val notification = NotificationCompat.Builder(this, channelId)
        .setSmallIcon(R.drawable.ic_launcher_foreground)
        .setContentTitle(title)
        .setContentText(description)
        .setContentIntent(pendingIntent)
        .setAutoCancel(true)
        .build()
    with(NotificationManagerCompat.from(this)) {
        if (ActivityCompat.checkSelfPermission(
                this@sendNotification,
                Manifest.permission.POST_NOTIFICATIONS
            ) != PackageManager.PERMISSION_GRANTED
        ) {
            return
        }
        notify(1, notification)
    }
}
