package dev.bg.bikebridge.util.ktx

import android.bluetooth.BluetoothGatt
import android.bluetooth.BluetoothGattCharacteristic
import android.bluetooth.BluetoothGattDescriptor

fun BluetoothGatt.getCharacteristics(): List<BluetoothGattCharacteristic> {
    if (services.isEmpty()) return emptyList()
    return services.flatMap { service -> service.characteristics }
}

fun BluetoothGatt.getGattTable(): String? {
    return if (services.isEmpty()) {
        null
    } else {
        services.joinToString(
            separator = "\n"
        ) { service ->
            "Service ${service.uuid}\n".plus(
                service.characteristics.joinToString(
                    separator = "\n|--",
                    prefix = "|--"
                ) { char ->
                    var description = "${char.uuid}: ${char.printProperties()}"
                    if (char.descriptors.isNotEmpty()) {
                        description += "\n" + char.descriptors.joinToString(
                            separator = "\n|-----",
                            prefix = "|-----"
                        ) { descriptor ->
                            "${descriptor.uuid}: ${descriptor.printProperties()}"
                        }
                    }
                    description
                }
            )
        }
    }
}

fun BluetoothGattCharacteristic.printProperties(): String = mutableListOf<String>().apply {
    if (isReadable()) add("READABLE")
    if (isWritable()) add("WRITABLE")
    if (isWritableWithoutResponse()) add("WRITABLE WITHOUT RESPONSE")
    if (isIndicatable()) add("INDICATABLE")
    if (isNotifiable()) add("NOTIFIABLE")
    if (isEmpty()) add("EMPTY")
}.joinToString()

fun BluetoothGattDescriptor.printProperties(): String = mutableListOf<String>().apply {
    if (isReadable()) add("READABLE")
    if (isWritable()) add("WRITABLE")
    if (isEmpty()) add("EMPTY")
}.joinToString()

fun BluetoothGattCharacteristic.isReadable(): Boolean =
    containsProperty(BluetoothGattCharacteristic.PROPERTY_READ)

fun BluetoothGattCharacteristic.isWritable(): Boolean =
    containsProperty(BluetoothGattCharacteristic.PROPERTY_WRITE)

fun BluetoothGattCharacteristic.isWritableWithoutResponse(): Boolean =
    containsProperty(BluetoothGattCharacteristic.PROPERTY_WRITE_NO_RESPONSE)

fun BluetoothGattCharacteristic.isIndicatable(): Boolean =
    containsProperty(BluetoothGattCharacteristic.PROPERTY_INDICATE)

fun BluetoothGattCharacteristic.isNotifiable(): Boolean =
    containsProperty(BluetoothGattCharacteristic.PROPERTY_NOTIFY)

fun BluetoothGattCharacteristic.containsProperty(property: Int): Boolean =
    properties and property != 0

fun BluetoothGattDescriptor.isReadable(): Boolean =
    containsPermission(BluetoothGattDescriptor.PERMISSION_READ)

fun BluetoothGattDescriptor.isWritable(): Boolean =
    containsPermission(BluetoothGattDescriptor.PERMISSION_WRITE)

fun BluetoothGattDescriptor.containsPermission(permission: Int): Boolean =
    permissions and permission != 0
