package dev.bg.bikebridge.util

import com.tencent.mmkv.MMKV
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update

private val kv = MMKV.defaultMMKV(MMKV.MULTI_PROCESS_MODE, null)

object Preferences {
    const val AUTO_SCAN = "as"
    const val AUTO_CONNECT = "ac"
    const val LAST_MAC = "lm"
    const val WRITE = "w"
    const val LOGGING = "log"
    const val CENSOR_LOG = "cl"
    const val GATT_TABLE = "gt"

    fun getBoolean(
        preference: String,
        defaultValue: Boolean = false
    ): Boolean = kv.decodeBool(preference, defaultValue)
    fun getString(
        preference: String
    ): String? = kv.decodeString(preference)

    fun setBoolean(
        preference: String,
        b: Boolean
    ) = kv.encode(preference, b);

    private fun setString(
        preference: String,
        s: String?
    ) = kv.encode(preference, s)

    fun setWrite(b: Boolean) {
        setBoolean(WRITE, b)
        _writeEnabledFlow.update { b }
    }

    fun setLogging(b: Boolean) {
        setBoolean(LOGGING, b)
        _logEnabledFlow.update { b }
    }

    fun setCensorLog(b: Boolean) {
        setBoolean(CENSOR_LOG, b)
        _censorLogFlow.update { b }
    }

    fun setLastMac(mac: String?) {
        setString(LAST_MAC, mac)
        _lastMacFlow.update { mac }
    }

    fun setAutoConnect(b: Boolean) {
        setBoolean(AUTO_CONNECT, b)
        _autoConnectFlow.update { b }
    }

    fun setAutoConnectFlow(b: Boolean) {
        _autoConnectFlow.update { b }
    }

    private val _writeEnabledFlow = MutableStateFlow(getBoolean(WRITE))
    val writeEnabledFlow = _writeEnabledFlow.asStateFlow()

    private val _logEnabledFlow = MutableStateFlow(getBoolean(LOGGING))
    val logEnabledFlow = _logEnabledFlow.asStateFlow()

    private val _censorLogFlow = MutableStateFlow(getBoolean(CENSOR_LOG))
    val censorLogFlow = _censorLogFlow.asStateFlow()

    private val _autoConnectFlow = MutableStateFlow(getBoolean(AUTO_CONNECT))
    val autoConnectFlow = _autoConnectFlow.asStateFlow()

    private val _lastMacFlow = MutableStateFlow(getString(LAST_MAC))
    val lastMacFlow = _lastMacFlow.asStateFlow()
}
