package dev.bg.bikebridge.util

import android.annotation.SuppressLint
import dev.bg.bikebridge.R
import dev.bg.bikebridge.ble.BleConstants
import dev.bg.bikebridge.util.ktx.xor
import javax.crypto.Cipher
import javax.crypto.spec.SecretKeySpec

// Payloads are little-endian
// null ret indicates this is probably the wrong tx/rx payload for that bike so disable writes
// some values are split across multiple bytes and need shifting
@OptIn(ExperimentalUnsignedTypes::class)
object BikeUtil {
    object Shimano {
        @SuppressLint("GetInstance")
        private fun aes(
            key: ByteArray,
            value: ByteArray
        ): ByteArray {
            val c = Cipher.getInstance("AES")
            c.init(Cipher.ENCRYPT_MODE, SecretKeySpec(key, "AES"))
            return c
                .doFinal(value)
                .take(16)
                .toByteArray()
        }

        fun getPasskeyAuthPayload(
            passkey: String,
            nonce: ByteArray
        ): ByteArray {
            return byteArrayOfInts(0x01) + aes((INIT_PRESHARED_IV + passkey.toByteArray()), nonce)
        }

        fun getAppAuthPayload(nonce: ByteArray): ByteArray {
            return byteArrayOfInts(0x02) + aes(PRESHARED_IV, nonce)
        }

        fun getMode(uBytes: UByteArray): BleConstants.Characteristics.Shimano.Mode {
            return when (uBytes[1]) {
                0x00.toUByte() -> BleConstants.Characteristics.Shimano.Mode.Off
                0x01.toUByte() -> BleConstants.Characteristics.Shimano.Mode.Eco
                0x02.toUByte() -> BleConstants.Characteristics.Shimano.Mode.Trail
                0x03.toUByte() -> BleConstants.Characteristics.Shimano.Mode.Boost
                0x04.toUByte() -> BleConstants.Characteristics.Shimano.Mode.Walk
                else -> BleConstants.Characteristics.Shimano.Mode.Unknown
            }
        }

        // km/h
        fun getSpeed(uBytes: UByteArray): Float {
            return (uBytes[2].toInt() or (uBytes[3].toInt() shl 8)).toFloat() / 10
        }

        // km
        // [1] | [2] << 8 | [3] << 16 | [4] << 24
        fun getTotalDistance(uBytes: UByteArray): Float {
            return (
                uBytes[1].toInt()
                or (uBytes[2].toInt() shl 8)
                or (uBytes[3].toInt() shl 16)
                or (uBytes[4].toInt() shl 24)
            ).toFloat() / 1000
        }

        // km/h
        fun getAverageSpeed(uBytes: UByteArray): Float {
            return (uBytes[9].toInt() or (uBytes[10].toInt() shl 8)).toFloat() / 10
        }

        // km/h
        fun getMaxSpeed(uBytes: UByteArray): Float {
            return (uBytes[11].toInt() or (uBytes[12].toInt() shl 8)).toFloat() / 10
        }

        fun isBeepEnabled(bytes: ByteArray): BleConstants.Characteristics.Shimano.Beep {
            return when (bytes[3]) {
                0x00.toByte() -> BleConstants.Characteristics.Shimano.Beep.Off
                0x01.toByte() -> BleConstants.Characteristics.Shimano.Beep.On
                else -> BleConstants.Characteristics.Shimano.Beep.Unknown
            }
        }

        fun getDisplayColor(bytes: ByteArray): BleConstants.Characteristics.Shimano.DisplayColor {
            return when (bytes[3]) {
                0x00.toByte() -> BleConstants.Characteristics.Shimano.DisplayColor.Black
                0x01.toByte() -> BleConstants.Characteristics.Shimano.DisplayColor.White
                else -> BleConstants.Characteristics.Shimano.DisplayColor.Unknown
            }
        }

        fun getDisplayLanguage(bytes: ByteArray): BleConstants.Characteristics.Shimano.DisplayLanguage {
            return when (bytes[3]) {
                0x00.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.English
                0x01.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.French
                0x02.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.German
                0x03.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.Dutch
                0x04.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.Italian
                0x05.toByte() -> BleConstants.Characteristics.Shimano.DisplayLanguage.Spanish
                else -> BleConstants.Characteristics.Shimano.DisplayLanguage.Unknown
            }
        }

        val INIT_PRESHARED_IV = listOf(128, 1048704, 17825920, 74449024, -95420288, 66060416, -98566016, 126877824, 85983360, 75497600).xor()
        val PRESHARED_IV = listOf(128, 2097280, -47185792, -104857472, -96468864, 39846016, -60817280, -113246080, -30408576, 11534464, -57671552, 35651712, -25165696, 128974976, -78643072, -120586112).xor()
    }

    object SRAM {
        fun getGear(uBytes: UByteArray): Int {
            return uBytes[5].toInt()
        }

        fun getMicroAdjust(uBytes: UByteArray): Int {
            return uBytes[8].toInt()
        }
    }

    fun isBike(id: Int) = bikeList.contains(id)

    fun getManufacturerResource(manufacturerId: Int): Int {
        return when (manufacturerId) {
            BleConstants.Manufacturers.BOSCH,
            BleConstants.Manufacturers.BOSCH_2 -> R.string.bosch
            BleConstants.Manufacturers.BROSE -> R.string.brose
            BleConstants.Manufacturers.DJI -> R.string.dji
            BleConstants.Manufacturers.FAZUA -> R.string.fazua
            BleConstants.Manufacturers.SHIMANO -> R.string.shimano
            BleConstants.Manufacturers.SRAM -> R.string.sram
            BleConstants.Manufacturers.TREK -> R.string.trek
            BleConstants.Manufacturers.YAMAHA -> R.string.yamaha
            else -> R.string.unknown_manufacturer
        }
    }

    private val bikeList = listOf(
        BleConstants.Manufacturers.BOSCH,
        BleConstants.Manufacturers.BOSCH_2,
        BleConstants.Manufacturers.BROSE,
        BleConstants.Manufacturers.DJI,
        BleConstants.Manufacturers.FAZUA,
        BleConstants.Manufacturers.SHIMANO,
        BleConstants.Manufacturers.SRAM,
        BleConstants.Manufacturers.TREK,
        BleConstants.Manufacturers.YAMAHA
    )
}
