package dev.bg.bikebridge.ui.screens.settings

import android.content.Intent
import android.net.Uri
import android.os.SystemClock
import android.provider.Settings
import androidx.compose.animation.AnimatedContent
import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.verticalScroll
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.ButtonDefaults
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.OutlinedButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableIntStateOf
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import dev.bg.bikebridge.BuildConfig
import dev.bg.bikebridge.R
import dev.bg.bikebridge.ui.components.SettingsSection
import dev.bg.bikebridge.ui.components.SettingsSwitch
import dev.bg.bikebridge.ui.components.SettingsTile
import dev.bg.bikebridge.util.Constants
import dev.bg.bikebridge.util.Preferences
import dev.bg.bikebridge.util.Preferences.setLogging
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlin.math.ceil

@Composable
fun SettingsScreen() {
    val ctx = LocalContext.current

    val write by Preferences.writeEnabledFlow.collectAsStateWithLifecycle()
    val logging by Preferences.logEnabledFlow.collectAsStateWithLifecycle()
    val censorLog by Preferences.censorLogFlow.collectAsStateWithLifecycle()

    var autoScan by remember { mutableStateOf(Preferences.getBoolean(Preferences.AUTO_SCAN, defaultValue = true)) }
    var autoConnect by remember { mutableStateOf(Preferences.getBoolean(Preferences.AUTO_CONNECT)) }
    var gattTable by remember { mutableStateOf(Preferences.getBoolean(Preferences.GATT_TABLE)) }

    Column(
        Modifier
            .fillMaxSize()
            .padding(horizontal = 16.dp)
            .verticalScroll(rememberScrollState())
    ) {
        Text(
            stringResource(R.string.settings),
            fontSize = 36.sp,
            fontWeight = FontWeight.W400
        )

        // Configuration
        SettingsSection(
            title = stringResource(R.string.configuration)
        ) {
            var writeWarning by remember { mutableStateOf(false) }

            SettingsSwitch(
                title = stringResource(R.string.auto_scan),
                description = stringResource(R.string.auto_scan_description),
                checked = autoScan,
                onChange = {
                    autoScan = !autoScan
                    Preferences.setBoolean(Preferences.AUTO_SCAN, autoScan)
                }
            )
            SettingsSwitch(
                title = stringResource(R.string.auto_connect),
                description = stringResource(R.string.auto_connect_description),
                checked = autoConnect,
                onChange = {
                    autoConnect = !autoConnect
                    Preferences.setAutoConnect(autoConnect)
                }
            )

            SettingsSwitch(
                title = stringResource(R.string.write_mode),
                description = stringResource(R.string.write_mode_description),
                checked = write,
                onChange = {
                    if (it) {
                        writeWarning = true
                    } else {
                        Preferences.setWrite(false)
                    }
                }
            )
            WriteModeWarning(
                visible = writeWarning,
                onDismissRequest = {
                    writeWarning = false
                },
                onConfirm = {
                    Preferences.setWrite(true)
                    writeWarning = false
                }
            )
        }
        HorizontalDivider()

        // Logging
        SettingsSection(
            title = stringResource(R.string.logging)
        ) {
            SettingsSwitch(
                title = stringResource(R.string.enable_logging),
                description = stringResource(R.string.enable_logging_description),
                checked = logging,
                onChange = {
                    setLogging(it)
                }
            )
            AnimatedVisibility(logging) {
                Column(
                    verticalArrangement = Arrangement.spacedBy(16.dp)
                ) {
                    SettingsSwitch(
                        title = stringResource(R.string.censor_log),
                        description = stringResource(R.string.censor_log_description),
                        checked = censorLog,
                        onChange = {
                            Preferences.setCensorLog(it)
                        }
                    )
                    SettingsSwitch(
                        title = stringResource(R.string.log_gatt_table),
                        description = stringResource(R.string.log_gatt_table_description),
                        checked = gattTable,
                        onChange = {
                            gattTable = !gattTable
                            Preferences.setBoolean(Preferences.GATT_TABLE, gattTable)
                        }
                    )
                }
            }
        }
        HorizontalDivider()

        // About
        SettingsSection(
            title = stringResource(R.string.about)
        ) {
            SettingsTile(
                title = stringResource(
                    R.string.about_name_template,
                    stringResource(R.string.app_name)
                ),
                description = stringResource(
                    R.string.about_package_name_template,
                    BuildConfig.APPLICATION_ID
                ),
                onClick = {
                    ctx.startActivity(
                        Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS).apply {
                            data = Uri.parse("package:${ctx.packageName}")
                        }
                    )
                }
            )
            SettingsTile(
                title = stringResource(R.string.about_version_template, BuildConfig.VERSION_NAME),
                description = stringResource(
                    R.string.about_build_number_template,
                    BuildConfig.VERSION_CODE
                )
            )
        }
    }
}

@Composable
private fun WriteModeWarning(
    visible: Boolean,
    onDismissRequest: () -> Unit,
    onConfirm: () -> Unit
) {
    if (visible) {
        val waitTime = 10000

        var remaining by remember { mutableIntStateOf(waitTime) }
        var ready by remember { mutableStateOf(false) }

        LaunchedEffect(Unit) {
            val start = SystemClock.uptimeMillis()
            while (isActive && remaining > 0) {
                remaining = (waitTime - ((SystemClock.uptimeMillis() - start).coerceAtLeast(0L))).coerceAtLeast(0L).toInt()
                if (remaining == 0) {
                    ready = true
                }
                delay(200.coerceAtMost(remaining).toLong())
            }
        }

        AlertDialog(
            onDismissRequest = onDismissRequest,
            title = {
                Text(stringResource(R.string.warning))
            },
            text = {
                Text(stringResource(R.string.write_warning_description))
//                BasicText(
//                    text = buildAnnotatedString {
//                        append(ctx.getString(R.string.write_warning_description))
//                        appendInlineContent("{{infoIcon}}", "{{infoIcon}}")
//                    },
//                    modifier = Modifier,
//                    inlineContent = mapOf(
//                        "{{infoIcon}}" to InlineTextContent(
//                            Placeholder(
//                                width = 12.sp,
//                                height = 12.sp,
//                                placeholderVerticalAlign = PlaceholderVerticalAlign.AboveBaseline
//                            )
//                        ) {
//                            Icon(
//                                Icons.Filled.Face,
//                                contentDescription = null
//                            )
//                        }
//                    )
//                )
            },
            dismissButton = {
                OutlinedButton(onClick = onDismissRequest) {
                    Text(stringResource(R.string.cancel))
                }
            },
            confirmButton = {
                Button(
                    enabled = remaining == 0,
                    colors = ButtonDefaults.buttonColors(
                        containerColor = Constants.Colors.ErrorRed,
                        contentColor = Color.White
                    ),
                    onClick = onConfirm
                ) {
                    AnimatedContent(ready, label = "") {
                        when (it) {
                            true -> {
                                Text(stringResource(R.string.enable))
                            }
                            false -> {
                                Text(ceil(remaining.toDouble() / 1000).toInt().toString())
                            }
                        }
                    }
                }
            }
        )
    }
}
