package dev.bg.bikebridge.ui.screens.scan

import android.annotation.SuppressLint
import android.bluetooth.le.ScanResult
import androidx.compose.animation.core.animateFloatAsState
import androidx.compose.foundation.ExperimentalFoundationApi
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxHeight
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.height
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.wrapContentHeight
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.material3.Button
import androidx.compose.material3.Card
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.DisposableEffect
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import dev.bg.bikebridge.LocalGattBinder
import dev.bg.bikebridge.LocalNavController
import dev.bg.bikebridge.R
import dev.bg.bikebridge.Route
import dev.bg.bikebridge.service.BluetoothGattService
import dev.bg.bikebridge.ui.components.LoadingPage
import dev.bg.bikebridge.ui.components.RADAR_SIZE_DP
import dev.bg.bikebridge.ui.components.Radar
import dev.bg.bikebridge.util.BikeUtil
import dev.bg.bikebridge.util.Preferences
import dev.bg.bikebridge.util.ktx.getManufacturerId
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.onEach
import timber.log.Timber

@OptIn(ExperimentalFoundationApi::class)
@Composable
fun ScanScreen() {
    val navController = LocalNavController.current
    val binder = LocalGattBinder.current

    binder ?: return

    val gattState = binder.state.collectAsStateWithLifecycle()

    var isConnectedToBike by remember { mutableStateOf(false) }

    val height = animateFloatAsState(if (gattState.value.scanning) RADAR_SIZE_DP else 0f, label = "")

    LaunchedEffect(Unit) {
        binder.events.onEach { event ->
            Timber.d("ScanScreen: received ${event.action} from service")
            when (event.action) {
                BluetoothGattService.GATT_CONNECTED -> {
                    isConnectedToBike = true
                    navController.navigate(Route.Bike)
                }
                BluetoothGattService.GATT_DISCONNECTED -> {
                    isConnectedToBike = false
                }
            }
        }.collect()
    }

    DisposableEffect(Unit) {
        if (Preferences.getBoolean(Preferences.AUTO_SCAN, defaultValue = false)) {
            binder.startScan()
        }

        onDispose {
            binder.stopScan()
        }
    }

    if (!isConnectedToBike) {
        Column(
            Modifier
                .fillMaxSize()
                .padding(16.dp)
                .padding(top = height.value.dp),
            verticalArrangement = Arrangement.SpaceBetween,
            horizontalAlignment = Alignment.CenterHorizontally
        ) {
            LazyColumn(
                Modifier
                    .fillMaxWidth()
                    .fillMaxHeight()
                    .weight(1f),
                verticalArrangement = Arrangement.spacedBy(16.dp)
            ) {
                if (gattState.value.bikesFound.isNotEmpty()) {
                    items(
                        gattState.value.bikesFound.toList(),
                        key = { bike -> bike.device.address }
                    ) { bike ->
                        BikeCard(
                            bike,
                            modifier = Modifier.animateItem(),
                            onClick = {
                                binder.connectToDevice(bike)
                            }
                        )
                    }
                }
            }
            Button(
                modifier = Modifier
                    .fillMaxWidth()
                    .wrapContentHeight(),
                onClick = {
                    if (gattState.value.scanning) {
                        binder.stopScan()
                    } else {
                        binder.startScan()
                    }
                }
            ) {
                if (gattState.value.scanning) {
                    Text(stringResource(R.string.stop_scan))
                } else {
                    Text(stringResource(R.string.start_scan))
                }
            }
        }
        if (gattState.value.scanning) {
            Box(Modifier.fillMaxSize(), contentAlignment = Alignment.TopCenter) {
                Radar()
            }
        }
    } else {
        LoadingPage()
    }
}

@SuppressLint("MissingPermission")
@Composable
fun BikeCard(
    scanResult: ScanResult,
    modifier: Modifier,
    onClick: () -> Unit
) {
    Card(
        modifier
            .height(90.dp)
            .fillMaxWidth()
            .clickable { onClick() },
        elevation = CardDefaults.elevatedCardElevation()
    ) {
        Row(
            Modifier
                .fillMaxWidth()
                .padding(8.dp),
            verticalAlignment = Alignment.CenterVertically,
            horizontalArrangement = Arrangement.SpaceBetween
        ) {
            Column(
                Modifier.fillMaxHeight(),
                verticalArrangement = Arrangement.Center,
                horizontalAlignment = Alignment.Start
            ) {
                val id = scanResult.getManufacturerId()
                if (id == null) {
                    Text(
                        scanResult.device.name,
                        fontSize = 24.sp
                    )
                } else {
                    Text(
                        scanResult.device.name,
                        fontSize = 24.sp
                    )
                    Text(stringResource(BikeUtil.getManufacturerResource(id)))
                }
            }
            Text("${scanResult.rssi}dBm")
        }
    }
}
