package dev.bg.bikebridge.ui.screens.bike.components

import android.bluetooth.BluetoothGattCharacteristic
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.Spacer
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.size
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.lazy.itemsIndexed
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Info
import androidx.compose.material3.AlertDialog
import androidx.compose.material3.Button
import androidx.compose.material3.HorizontalDivider
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.saveable.rememberSaveable
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import dev.bg.bikebridge.LocalGattBinder
import dev.bg.bikebridge.R
import dev.bg.bikebridge.ble.BleConstants
import dev.bg.bikebridge.data.GattServiceEventAction
import dev.bg.bikebridge.data.ShimanoSettingsNotification
import dev.bg.bikebridge.data.state.GattServiceState
import dev.bg.bikebridge.service.BluetoothGattService
import dev.bg.bikebridge.ui.components.BikeSettingsOptionDialog
import dev.bg.bikebridge.ui.components.BikeSettingsSegmentedButtons
import dev.bg.bikebridge.ui.components.BikeSettingsSwitch
import dev.bg.bikebridge.ui.components.MonospaceText
import dev.bg.bikebridge.util.BikeUtil
import dev.bg.bikebridge.util.ktx.assertivelyGet
import dev.bg.bikebridge.util.ktx.getShortUuid
import dev.bg.bikebridge.util.ktx.startsWith
import dev.bg.bikebridge.util.ktx.toHexString
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.onEach
import timber.log.Timber

@Composable
fun ShimanoSettings(
    state: GattServiceState
) {
    val ctx = LocalContext.current
    val binder = LocalGattBinder.current

    binder ?: return

    var initialised by rememberSaveable { mutableStateOf(false) }
    var triggerPayloadDialog by remember { mutableStateOf(false) }

    // settings
    var beep by rememberSaveable { mutableStateOf<BleConstants.Characteristics.Shimano.Beep?>(null) } // TODO these can't be saveable
    var beepWait by rememberSaveable { mutableStateOf(true) }
    var displayColor by rememberSaveable { mutableStateOf<BleConstants.Characteristics.Shimano.DisplayColor?>(null) }
    var displayColorWait by rememberSaveable { mutableStateOf(true) }
    var displayLanguage by rememberSaveable { mutableStateOf<BleConstants.Characteristics.Shimano.DisplayLanguage?>(null) }
    var displayLanguageWait by rememberSaveable { mutableStateOf(true) }

    LaunchedEffect(Unit) {
        binder.events.onEach { event ->
            when (event) {
                is GattServiceEventAction -> {
                    if (event.action == BluetoothGattService.GATT_ENQUEUED_TRIGGERS) {
                        initialised = true
                    }
                }
                is ShimanoSettingsNotification -> {
                    when {
                        // beep
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.BEEP_FILTER) ||
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.BEEP_TRIGGER_FILTER) -> {
                            val newBeep = BikeUtil.Shimano.isBeepEnabled(event.payload)
                            beepWait = false
                            beep = newBeep
                            Timber.d("Beep: ${ctx.getString(newBeep.res)}")
                        }
                        // display color
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.DISPLAY_COLOR_FILTER) ||
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.DISPLAY_COLOR_TRIGGER_FILTER) -> {
                            val color = BikeUtil.Shimano.getDisplayColor(event.payload)
                            displayColorWait = false
                            displayColor = color
                            Timber.d("Display color: ${ctx.getString(color.res)}")
                        }
                        // display language
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.DISPLAY_LANGUAGE_FILTER) ||
                        event.payload.startsWith(BleConstants.Characteristics.Shimano.Write.Settings.DISPLAY_LANGUAGE_TRIGGER_FILTER) -> {
                            val language = BikeUtil.Shimano.getDisplayLanguage(event.payload)
                            displayLanguageWait = false
                            displayLanguage = language
                            Timber.d("Language: ${ctx.getString(language.res)}")
                        }
                    }
                }
            }
        }.collect()
    }

    Column(
        Modifier.fillMaxWidth(),
        verticalArrangement = Arrangement.spacedBy(12.dp)
    ) {
        Spacer(Modifier.size(6.dp))
        // 10 bytes
        Text(
            stringResource(R.string.settings),
            fontSize = 24.sp,
            fontWeight = FontWeight.W400
        )
        when {
            !initialised && state.writableRequiresTrigger == true -> { // TODO && !autoTxTriggers (settings option)
                Box(
                    Modifier
                        .fillMaxWidth()
                        .padding(horizontal = 32.dp),
                    contentAlignment = Alignment.Center
                ) {
                    Column(
                        verticalArrangement = Arrangement.spacedBy(16.dp),
                        horizontalAlignment = Alignment.CenterHorizontally
                    ) {
                        Text(
                            stringResource(R.string.trigger_description),
                            textAlign = TextAlign.Center
                        )
                        Row {
                            IconButton(onClick = { triggerPayloadDialog = true }) {
                                Icon(
                                    Icons.Filled.Info,
                                    contentDescription = stringResource(R.string.payload_info)
                                )
                            }
                            Button(
                                onClick = {
                                    binder.txTriggers()
                                }
                            ) {
                                Text(stringResource(R.string.enable))
                            }
                        }
                    }
                }
            }
            initialised -> {
                if (state.writableCharacteristics.any { c -> c.uuid == BleConstants.Characteristics.Shimano.SETTINGS_COMMAND }) {
                    BikeSettingsSwitch(
                        title = stringResource(R.string.beep),
                        options = listOf(
                            BleConstants.Characteristics.Shimano.Beep.Off,
                            BleConstants.Characteristics.Shimano.Beep.On
                        ),
                        checked = beep == BleConstants.Characteristics.Shimano.Beep.On,
                        enabled = beep != null,
                        waiting = beepWait,
                        onChange = {
                            beepWait = true
                            binder.writeCharacteristic(
                                BleConstants.Characteristics.Shimano.SETTINGS_COMMAND,
                                if (it) {
                                    BleConstants.Characteristics.Shimano.Beep.On.payload!!
                                } else {
                                    BleConstants.Characteristics.Shimano.Beep.Off.payload!!
                                }
                            )
                        }
                    )
                    BikeSettingsSegmentedButtons(
                        title = stringResource(R.string.display_color),
                        options = listOf(
                            BleConstants.Characteristics.Shimano.DisplayColor.Black,
                            BleConstants.Characteristics.Shimano.DisplayColor.White
                        ),
                        selected = displayColor,
                        enabled = displayColor != null,
                        waiting = displayColorWait,
                        onClick = {
                            displayColorWait = true
                            binder.writeCharacteristic(
                                BleConstants.Characteristics.Shimano.SETTINGS_COMMAND,
                                if (it.payload != null) {
                                    it.payload!!
                                } else {
                                    return@BikeSettingsSegmentedButtons
                                }
                            )
                        }
                    )
                    BikeSettingsOptionDialog(
                        title = stringResource(R.string.display_language),
                        options = listOf(
                            BleConstants.Characteristics.Shimano.DisplayLanguage.English,
                            BleConstants.Characteristics.Shimano.DisplayLanguage.French,
                            BleConstants.Characteristics.Shimano.DisplayLanguage.German,
                            BleConstants.Characteristics.Shimano.DisplayLanguage.Dutch,
                            BleConstants.Characteristics.Shimano.DisplayLanguage.Italian,
                            BleConstants.Characteristics.Shimano.DisplayLanguage.Spanish
                        ),
                        selected = displayLanguage,
                        waiting = displayLanguageWait,
                        onSelected = {
                            displayLanguageWait = true
                            binder.writeCharacteristic(
                                BleConstants.Characteristics.Shimano.SETTINGS_COMMAND,
                                if (it.payload != null) {
                                    it.payload!!
                                } else {
                                    return@BikeSettingsOptionDialog
                                }
                            )
                        }
                    )
                }
            }
        }
    }
    TriggerPayloadDialog(
        visible = triggerPayloadDialog,
        writableCharacteristics = state.writableCharacteristics,
        onDismissRequest = { triggerPayloadDialog = false }
    )
}

@Composable
private fun TriggerPayloadDialog(
    visible: Boolean,
    writableCharacteristics: List<BluetoothGattCharacteristic>,
    onDismissRequest: () -> Unit
) {
    if (visible) {
        AlertDialog(
            onDismissRequest = onDismissRequest,
            confirmButton = {
                Button(onClick = onDismissRequest) {
                    Text(stringResource(R.string.ok))
                }
            },
            title = { Text(stringResource(R.string.payload_info)) },
            text = {
                LazyColumn(
                    Modifier.fillMaxWidth(),
                    verticalArrangement = Arrangement.spacedBy(4.dp)
                ) {
                    item {
                        Row(
                            Modifier.padding(bottom = 4.dp),
                            horizontalArrangement = Arrangement.SpaceBetween
                        ) {
                            Text("UUID")
                            Spacer(Modifier.weight(1f))
                            Text(stringResource(R.string.payload))
                        }
                        HorizontalDivider()
                    }
                    items(
                        writableCharacteristics.filter { c -> BleConstants.TRIGGERS.contains(c.uuid) }
                    ) { c ->
                        LazyColumn(
                            Modifier.fillMaxWidth(),
                            verticalArrangement = Arrangement.spacedBy(4.dp),
                            userScrollEnabled = false
                        ) {
                            itemsIndexed(
                                BleConstants.TRIGGER_PAYLOAD_MAP.assertivelyGet(c.uuid)
                            ) { idx, payload ->
                                Row(
                                    horizontalArrangement = Arrangement.SpaceBetween
                                ) {
                                    if (idx == 0) {
                                        MonospaceText(c.getShortUuid())
                                    }
                                    Spacer(Modifier.weight(1f))
                                    MonospaceText(
                                        payload.toHexString(),
                                        textAlign = TextAlign.End
                                    )
                                }
                            }
                        }
                    }
                }
            }
        )
    }
}
