package dev.bg.bikebridge.ui.screens.bike.components

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.tween
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.size
import androidx.compose.material3.Icon
import androidx.compose.material3.Text
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.Path
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.res.painterResource
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import dev.bg.bikebridge.R
import dev.bg.bikebridge.util.Constants
import kotlin.math.absoluteValue

@Composable
fun DeviceInfo(
    deviceName: String?,
    manufacturer: String,
    firmwareVersion: String?,
    rssi: Int,
    bonded: Boolean
) {
    Row(
        Modifier.fillMaxWidth(),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.SpaceBetween
    ) {
        Column(
            modifier = Modifier.weight(0.6f),
            verticalArrangement = Arrangement.spacedBy(8.dp)
        ) {
            Text(
                deviceName ?: stringResource(R.string.bike),
                fontSize = 36.sp,
                fontWeight = FontWeight.W400
            )
            Text(manufacturer)
            AnimatedVisibility(firmwareVersion != null) {
                if (!firmwareVersion.isNullOrEmpty()) {
                    Text(stringResource(R.string.firmware_version, firmwareVersion))
                }
            }
        }
        Column(
            modifier = Modifier.weight(0.4f),
            verticalArrangement = Arrangement.spacedBy(8.dp),
            horizontalAlignment = Alignment.End
        ) {
            Row(
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.spacedBy(8.dp)
            ) {
                AnimatedVisibility(rssi != 0) {
                    val factor = remember { Animatable(0f) }

                    LaunchedEffect(rssi) {
                        factor.animateTo(
                            targetValue = ((1f / (rssi.toFloat().absoluteValue / 110f)) - 1f).absoluteValue.coerceIn(0.01f, 1f),
                            animationSpec = tween(1500)
                        )
                    }

                    Canvas(Modifier.size(20.dp)) {
                        val w = size.width * factor.value

                        drawPath(
                            Path().apply {
                                moveTo(w, size.height * (1 - factor.value))
                                lineTo(w, size.height)
                                lineTo(0f, size.height)
                            },
                            color = when {
                                factor.value <= 0.33f -> Constants.IndicatorColors.Low
                                0.33f < factor.value && factor.value <= 0.66f -> Constants.IndicatorColors.Medium
                                0.66f < factor.value -> Constants.IndicatorColors.High
                                else -> Constants.IndicatorColors.Low
                            }
                        )
                        drawPath(
                            Path().apply {
                                moveTo(size.width, 0f)
                                lineTo(size.width, size.height) // v
                                lineTo(0f, size.height) // h
                                lineTo(size.width, 0f) // c
                            },
                            color = Color.Gray,
                            style = Stroke(5f)
                        )
                    }
                }
                Text(
                    "$rssi dBm",
                    fontSize = 24.sp
                )
            }
            Row(
                verticalAlignment = Alignment.CenterVertically,
                horizontalArrangement = Arrangement.spacedBy(4.dp)
            ) {
                Icon(
                    if (bonded) painterResource(R.drawable.baseline_bluetooth_connected_24) else painterResource(R.drawable.baseline_bluetooth_disabled_24),
                    contentDescription = null
                )
                Text(if (bonded) stringResource(R.string.bonded) else stringResource(R.string.not_bonded))
            }
        }
    }
}
